import logging
import re

from django.conf import settings
from rest_framework.response import Response

from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.cloudsearch.exceptions import FormatterIsDownError
from wiki.cloudsearch.utils import find_by_search_uuid, render_document_for_indexation
from wiki.api_core.utils import TVMIDPermission
from wiki.pages.models import Page
from wiki.sync.connect.org_ctx import org_ctx
from wiki.utils.wfaas.client import get_wfaas_client
from wiki.utils.wfaas.errors import WfaasApiRequestError

logger = logging.getLogger(__name__)
wfaas_client = get_wfaas_client()


class TVMPermission(TVMIDPermission):
    tvm_client_ids = [settings.CLOUDSEARCH_TVM_CLIENT_ID]
    strict = True  # Пока в эту ручку ходят сервисы по OAuth не отрубаем запросы которые авторизованы не по TVM2


class GetDocument(WikiAPIView):
    permission_classes = (TVMPermission,)  # Нужно, чтобы клиент микросервиса мог дерать эту ручку

    @raises()
    def post(self, request, *args, **kwargs):
        indexable_model = find_by_search_uuid(request.data['uuid'])
        with org_ctx(indexable_model.get_model_org()):
            data = render_document_for_indexation(indexable_model)
            logger.info('document rendered successfully')

            if 'body' in data['document']:
                to_html_method = wfaas_client.raw_to_html
                if data['metadata']['page_type'] == Page.TYPES.WYSIWYG:
                    to_html_method = wfaas_client.yfm_to_html

                try:
                    data['document']['body'] = to_html_method(data['document']['body'])
                except WfaasApiRequestError:
                    raise FormatterIsDownError()

                data['document']['body'] = re.sub(
                    '{{ ?(a|anchor|forms|iframe|grid|tree|toc|mypages|mywatches|backlink|tasks).*?}}',
                    ' ',
                    data['document']['body'],
                )
            logger.info('document body formatted successfully')

            return Response(data, 200)
