import logging
from typing import Dict

import copy
from rest_framework.response import Response
from wiki.api_core.errors.rest_api_error import RestApiError
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.query_view import QueryViewSerializer
from wiki.cloudsearch.cloudsearch_client import CloudSearchClient
from wiki.cloudsearch.tasks import IndexOrganizationTask
from wiki.org import get_org
from wiki.sync.connect.models import Organization
from wiki.utils import lock

logger = logging.getLogger(__name__)


class QueryView(WikiAPIView):
    serializer_class = QueryViewSerializer

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Example: https://wiki-api.yandex-team.ru/_api/frontend/.search?q=hello&page_id=1
        """
        org = get_org()

        data = {
            'fuzzy': request.GET.get('fuzzy', False),
            'query': request.GET.get('q', ''),
            'page_id': request.GET.get('page_id', 1),
            'page_size': request.GET.get('page_size', 5),
            'acl': {
                'org_id': org.dir_id,
                'user': {'uid': request.user.get_uid(), 'cloud_uid': request.user.get_cloud_uid()},
                'group_ids': request.user.get_all_group_ids(),
            },
        }

        if not check_org_is_indexed(org):
            raise AlreadyIndexingError()

        extracted_response = CloudSearchClient.query_view(data)
        return Response(extracted_response)

    @raises()
    def post(self, request, *args, **kwargs):
        org = get_org()

        data = request.data

        serializer = self.get_serializer(data=change_names(data))
        serializer.is_valid(raise_exception=True)
        data['acl'] = {
            'org_id': org.dir_id,
            'user': {'uid': request.user.get_uid(), 'cloud_uid': request.user.get_cloud_uid()},
            'group_ids': request.user.get_all_group_ids(),
        }

        if not check_org_is_indexed(org):
            raise AlreadyIndexingError()

        extracted_response = CloudSearchClient.query_view(data)
        return Response(extracted_response)


def change_names(data: Dict):
    # Это сделано, потому что masstermax (я) зачем-то придумал передавать айдишники через authors.uid
    data_ = copy.deepcopy(data)
    for filter_name in data.get('filters', {}):
        if '.' in filter_name:
            new_name = filter_name.replace('.', '_')
            data_['filters'][new_name] = data_['filters'].pop(filter_name)
        elif filter_name in {'created_at', 'modified_at'} and 'from' in data_['filters'][filter_name]:
            data_['filters'][filter_name]['from_'] = data_['filters'][filter_name].pop('from')
    return data_


def check_org_is_indexed(org: Organization):
    if org.is_indexed:
        return True

    lock_name = f'index_organization_{org.id}'

    try:
        if not lock.check_lock_acquired(lock_name):
            logger.info(f'send indexing org: {org.id} to celery')
            IndexOrganizationTask().delay(org_id=org.id)
    except lock.LockTaken:
        logger.info('Celery task sending refused')
    except lock.FailedToGetLockError:
        logger.exception('Failed to get pre-lock')

    return False


class AlreadyIndexingError(RestApiError):
    status_code = 400
    error_code = 'ALREADY_INDEXING'
    debug_message = 'current organization indexing in progress'
