import logging

from django.contrib.auth import get_user_model
from django.db import IntegrityError, transaction
from itertools import chain
from typing import List

from wiki.api_v2.public.me.favorites.exceptions import (
    BookmarkNotFound,
    TagNotFound,
)

from wiki.api_v2.exceptions import AlreadyExists, Forbidden
from wiki.favorites.consts import AutoBookmarkType
from wiki.favorites.models import AutoBookmark, Bookmark, Tag
from wiki.pages.models import Page
from wiki.sync.connect.base_organization import BaseOrganization

AUTOBOOKMARK_LIMIT = 250  # Ограничение на каждый из типов, суммарно будет 500

User = get_user_model()
logger = logging.getLogger(__name__)


def get_user_bookmarks(user: User, organization: BaseOrganization, tags: List[str] = None):
    bookmarks = (
        Bookmark.objects.filter(
            user=user,
            page__org=organization.as_django_model(),
        )
        .select_related('page')
        .prefetch_related('tags')
        .order_by('-created_at')
    )

    if tags:
        bookmarks = bookmarks.filter(tags__name__in=tags).distinct()

    return bookmarks


def get_user_autobookmarks(user: User, organization: BaseOrganization, bookmark_type: AutoBookmarkType = None):
    bookmarks = (
        AutoBookmark.objects.filter(
            user=user,
            page__org=organization.as_django_model(),
        )
        .select_related('page')
        .order_by('-created_at')
    )
    if bookmark_type is not None:
        bookmarks = bookmarks.filter(bookmark_type=bookmark_type)
    return bookmarks


@transaction.atomic
def create_bookmark(user: User, organization: BaseOrganization, page: Page, tags: List[str]):
    org = organization.as_django_model()
    if not page.org == org:
        raise Forbidden('org mismatch')

    try:
        bookmark = Bookmark.objects.create(
            user=user,
            page=page,
        )
    except IntegrityError:
        raise AlreadyExists()

    if tags:
        update_bookmark_tags(user, organization, bookmark, tags)

    return bookmark


@transaction.atomic
def update_bookmark(bookmark_id: int, user: User, organization: BaseOrganization, tags: List[str]):
    org = organization.as_django_model()
    try:
        bookmark = Bookmark.objects.get(id=bookmark_id, user=user, page__org=org)
    except Bookmark.DoesNotExist:
        raise BookmarkNotFound()

    update_bookmark_tags(user, organization, bookmark, tags)
    return bookmark


def delete_bookmark(bookmark_id: int, user: User, organization: BaseOrganization):
    org = organization.as_django_model()
    try:
        bookmark = Bookmark.objects.get(id=bookmark_id, user=user, page__org=org)
    except Bookmark.DoesNotExist:
        raise BookmarkNotFound()

    bookmark.delete()


@transaction.atomic
def update_bookmark_tags(user: User, organization: BaseOrganization, bookmark: Bookmark, tags: List[str]):
    existing_tags, created_tags = _create_tags(user, organization, tags)

    united_tags = [tag for tag in chain(existing_tags, created_tags)]
    bookmark.tags.add(*united_tags)

    # Удаляем теги, которые уже есть на закладке, но которых нет в переданном списке
    extra_tags = bookmark.tags.exclude(name__in=tags)
    if extra_tags:
        bookmark.tags.remove(*extra_tags)


def _create_tags(user: User, organization: BaseOrganization, tags: List[str]):
    org = organization.as_django_model()
    existing_tags = Tag.objects.filter(user=user, org=org, name__in=tags).order_by('name')

    missed_tags = set(tags) - {tag.name for tag in existing_tags}
    created_tags = []
    if missed_tags:
        try:
            created_tags = Tag.objects.bulk_create(
                [Tag(name=tag_name, user=user, org=org) for tag_name in sorted(missed_tags)]
            )
        except IntegrityError:
            raise AlreadyExists()

    return existing_tags, created_tags


@transaction.atomic
def create_tags(user: User, organization: BaseOrganization, tags: List[str]):
    _, created_tags = _create_tags(user, organization, tags)
    return created_tags


def delete_tags(user: User, organization: BaseOrganization, tags: List[str]):
    org = organization.as_django_model()
    existing_tags = Tag.objects.filter(name__in=tags, user=user, org=org)

    if len(tags) > existing_tags.count():
        raise TagNotFound('some tags not found')

    existing_tags.delete()


def update_autobookmarks(user_id: int, page_id: int, is_new_page: bool):
    try:
        page = Page.objects.get(id=page_id)
    except Page.DoesNotExist:
        logger.error(f'Page with id {page_id} does not exist')
        return

    if not page.status > 0:
        # Страницу уже успели удалить, не добавляем её в закладки
        return

    try:
        # Не проверяем тип закладки чтобы не дублировать одну и ту же страницу в разных группах
        AutoBookmark.objects.get(user_id=user_id, page=page)
    except AutoBookmark.DoesNotExist:
        bookmark_type = AutoBookmarkType.CREATOR if is_new_page else AutoBookmarkType.EDITOR
        try:
            AutoBookmark.objects.create(user_id=user_id, page_id=page_id, bookmark_type=bookmark_type)
        except IntegrityError:
            return  # Если ничего не создали, то дальше и обрезать ничего не надо

        # Oбрезаем все, что больше лимита
        last_ids = (
            AutoBookmark.objects.filter(
                user_id=user_id,
                page__org=page.org,
                bookmark_type=bookmark_type,
            )
            .order_by('-created_at')
            .values_list('id', flat=True)[:AUTOBOOKMARK_LIMIT]
        )

        AutoBookmark.objects.filter(
            user_id=user_id,
            page__org=page.org,
            bookmark_type=bookmark_type,
        ).exclude(id__in=last_ids).delete()
