import logging

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand, CommandError

from wiki.favorites_v2.models import Folder
from wiki.favorites.consts import AutoBookmarkType
from wiki.favorites.models import AutoBookmark, Bookmark, Tag
from wiki.favorites.logic import create_bookmark
from wiki.org import get_user_orgs
from wiki.pages.models import Page
from wiki.sync.connect.base_organization import as_base_organization
from wiki.sync.connect.models import Organization
from wiki.sync.connect.org_ctx import org_ctx
from wiki.users.logic.settings import uses_new_favorites, set_user_setting
from wiki.utils.models import queryset_iterator

logger = logging.getLogger(__name__)
User = get_user_model()


class Command(BaseCommand):
    def add_arguments(self, parser):
        parser.add_argument('--full', type=bool)
        parser.add_argument('--force', type=bool)
        parser.add_argument('--username', type=str)
        parser.add_argument('--org_id', type=int)

    def handle(*args, **options):
        force = options.get('force', False)
        if options.get('full'):
            do_full_sync(force)
            return

        username = options.get('username')
        org_id = options.get('org_id')

        if not username:
            raise CommandError('empty username')

        org = None
        if settings.IS_BUSINESS:
            if not org_id:
                raise CommandError('empty org_id')

            try:
                org = Organization.objects.get(id=org_id)
            except Organization.DoesNotExist:
                raise CommandError(f'organization {org_id} does not found')

        try:
            user_filter = {'username': username}
            if hasattr(User, 'orgs'):
                user_filter['orgs'] = org
            user = User.objects.get(**user_filter)
        except User.DoesNotExist:
            raise CommandError(f'user {username} does not found')

        do_sync(user, org, force)


def do_full_sync(force=False):
    users_qs = get_user_model().objects.filter(is_active=True)
    for user in queryset_iterator(users_qs, 5000):
        sync_user_favorites(user, force)


def sync_user_favorites(user, force=False):
    for org in get_user_orgs(user, assert_empty_org=False):
        do_sync(user, org, force)


def do_sync(user, org, force=False):
    if uses_new_favorites(user) and not force:
        logger.warning(f'user {user.username} has already synced')
        return

    sync_user_bookmarks(user, org)
    sync_user_autobookmarks(user, org)
    with org_ctx(org):
        # В б2б на сохранение пользователя сейчас висят сигналы
        # которые создают дефолтные папки для старого избранного
        # для них нужен контекст организации
        set_user_setting(user, 'new_favorites', True)


def sync_user_bookmarks(user, org=None):
    logger.info(f'sync bookmark for {user.username} org {org}')
    folders = Folder.objects.filter(user=user, org=org).exclude(type=Folder.FOLDER_TYPE_AUTO)
    for folder in folders:
        logger.info(f'sync folder {folder.name}')

        folder_name = folder.name[:50]
        if folder_name != folder.name:
            logger.info(f'rename folder to {folder_name}')

        if folder.type == Folder.FOLDER_TYPE_CUSTOM:
            Tag.objects.get_or_create(name=folder_name, user=user, org=org)
        for bookmark in folder.bookmark_set.all().order_by('created_at'):
            if bookmark.supertag:
                try:
                    page = Page.active.get(supertag=bookmark.supertag, org=org)
                except Page.DoesNotExist:
                    logger.warning(f'cannot find page for supertag {bookmark.supertag}')
                    continue
                already_exists = Bookmark.objects.filter(user=user, page=page).exists()
                if not already_exists:
                    tags = [folder_name] if folder.type == Folder.FOLDER_TYPE_CUSTOM else []
                    create_bookmark(user, as_base_organization(org), page, tags)


def sync_user_autobookmarks(user, org=None):
    logger.info(f'sync auto bookmark for {user.username} org {org}')
    folders = Folder.objects.filter(
        user=user,
        org=org,
        name__in=[Folder.OWNER_AUTOFOLDER_NAME, Folder.LAST_EDIT_AUTOFOLDER_NAME]
    )
    for folder in folders:
        logger.info(f'sync folder {folder.name}')
        for bookmark in folder.autobookmark_set.all().order_by('created_at'):
            if bookmark.supertag:
                try:
                    page = Page.active.get(supertag=bookmark.supertag, org=org)
                except Page.DoesNotExist:
                    logger.warning(f'cannot find page for supertag {bookmark.supertag}')
                    continue
                already_exists = AutoBookmark.objects.filter(user=user, page=page).exists()
                if not already_exists:
                    if folder.name == Folder.OWNER_AUTOFOLDER_NAME:
                        bookmark_type = AutoBookmarkType.CREATOR
                        created_at = page.created_at
                    elif folder.name == Folder.LAST_EDIT_AUTOFOLDER_NAME:
                        bookmark_type = AutoBookmarkType.EDITOR
                        created_at = bookmark.created_at
                    else:
                        raise ValueError(f'Unknown folder name {folder.name}')

                    AutoBookmark.objects.create(
                        user=user,
                        page=page,
                        bookmark_type=bookmark_type,
                        created_at=created_at,
                    )
