from collections import defaultdict
from itertools import chain

from django.db.models import F
from django.utils.translation import ugettext_lazy

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.favorites_v2.models import AutoBookmark, Bookmark, Folder
from wiki.favorites.consts import AutoBookmarkType
from wiki.favorites.logic import update_bookmark_tags
from wiki.favorites.models import AutoBookmark as NewAutoBookmark, Bookmark as NewBookmark
from wiki.org import get_org
from wiki.sync.connect.base_organization import as_base_organization


def get_folder(folder_name, user):
    """
    Проверить, существует ли папка с запрашиваемым folder_name в базе или нет у данного пользователя.
    """
    try:
        return Folder.objects.get(user=user, name=folder_name, org=get_org())
    except Folder.DoesNotExist:
        return None


def update_bookmarks_indexes(folder):
    """
    Увеличить индекс всех закладок в переданной папке на единицу.
    """
    Bookmark.objects.filter(folder_id=folder.id).update(index_number=F('index_number') + 1)


def get_or_create_folder(user, folder_name):
    """
    Найти или создать папку запрашиваемым folder_name для указанного пользователя.
    """
    folder = get_folder(folder_name, user)
    if not folder:
        folders_indexes = (
            Folder.objects.filter(
                user=user,
                type=Folder.FOLDER_TYPE_CUSTOM,
                org=get_org(),
            )
            .order_by('-index_number')
            .values_list('index_number', flat=True)
        )

        next_folder_index = folders_indexes[0] + 1 if folders_indexes else 1
        folder = Folder(
            name=folder_name,
            favorites_count=0,
            user=user,
            index_number=next_folder_index,
            type=Folder.FOLDER_TYPE_CUSTOM,
            org=get_org(),
        )
        folder.save()

    return folder


def create_bookmark(folder, title, url):
    # Проверяем, что такой закладки еще нет
    if Bookmark.objects.filter(folder=folder, url=url).exists():
        raise InvalidDataSentError(ugettext_lazy('Bookmark already exists'))

    bookmark = Bookmark(folder=folder, index_number=0 if folder.favorites_count else 1, title=title, url=url)

    from wiki.api_core.utils import find_wiki_page_by_url

    bookmark.supertag, page = find_wiki_page_by_url(bookmark.url)
    if page:
        if page.last_author:
            bookmark.page_last_editor = page.last_author.username

        bookmark.page_modified_at = page.modified_at

    folder.favorites_count += 1
    folder.save()
    bookmark.save()

    if folder.favorites_count > 1:
        update_bookmarks_indexes(folder)

    return bookmark


def create_bookmark_compat(user, url, tag):
    from wiki.api_core.utils import find_wiki_page_by_url
    _, page = find_wiki_page_by_url(url)
    if not page:
        raise InvalidDataSentError(ugettext_lazy(f"Can't find page for url {url}"))

    if NewBookmark.objects.filter(user=user, page=page).exists():
        raise InvalidDataSentError(ugettext_lazy('Bookmark already exists'))

    tags = [] if tag in Folder.RESERVED_FOLDER_NAMES_LIST else [tag]
    bookmark = NewBookmark.objects.create(user=user, page=page)
    update_bookmark_tags(user, as_base_organization(get_org()), bookmark, tags)
    return bookmark


def get_user_bookmarks_by_folders(user):
    """
    Вернуть все закладки пользователя, сгруппированные по папкам.
    Ищем закладки во всех папках пользователя, включая автопапки.

    @return словарь, где ключ - имя папки, значение - список закладок в папке
    (элементы списка - объекты типа Bookmark или AutoBookmark)
    @rtype dict
    """
    bookmarks = (
        Bookmark.objects.filter(folder__user=user, folder__org=get_org())
        .order_by('folder__id')
        .select_related('folder')
    )
    autobookmarks = (
        AutoBookmark.objects.filter(folder__user=user, folder__org=get_org())
        .order_by('folder__id', '-page_modified_at')
        .select_related('folder')
    )

    result = dict()
    for bookmark in chain(bookmarks, autobookmarks):
        if bookmark.folder.name in result:
            result[bookmark.folder.name].append(bookmark)
        else:
            result[bookmark.folder.name] = [bookmark]

    return result


def get_user_bookmarks_by_folders_compat(user):
    result = defaultdict(list)

    bookmarks = (
        NewBookmark.objects
        .filter(user=user, page__org=get_org())
        .select_related('page')
        .prefetch_related('tags')
        .order_by('-created_at')
    )
    for bookmark in bookmarks:
        tags = bookmark.tags.all()
        if tags:
            for tag in bookmark.tags.all():
                result[tag.name].append(bookmark)
        else:
            result[Folder.FAVORITES_FOLDER_NAME].append(bookmark)

    autobookmarks = (
        NewAutoBookmark.objects.filter(
            user=user,
            page__org=get_org(),
        )
        .select_related('page')
        .order_by('-created_at')
    )

    folder_by_type = {
        AutoBookmarkType.CREATOR: Folder.OWNER_AUTOFOLDER_NAME,
        AutoBookmarkType.EDITOR: Folder.LAST_EDIT_AUTOFOLDER_NAME,
    }

    for bookmark in autobookmarks:
        folder_name = folder_by_type[bookmark.bookmark_type]
        result[folder_name].append(bookmark)

    # В новой схеме нет аналога папки __WATCHER__
    # берем данные из старой схемы
    watched_bookmarks = (
        AutoBookmark.objects.filter(
            folder__user=user,
            folder__org=get_org(),
            folder__name=Folder.WATCHER_AUTOFOLDER_NAME,
        )
        .order_by('folder__id', '-page_modified_at')
        .select_related('folder')
    )
    for bookmark in watched_bookmarks:
        result[bookmark.folder.name].append(bookmark)

    return result
