
from django.db import models
from django.conf import settings

from wiki.sync.connect.models import Organization
from wiki.org import get_org


class Folder(models.Model):
    """
    Модель папки в пользовательском меню (Избранном).
    """

    # Имя-метка для папки Избранное, хранящееся в базе данных.
    FAVORITES_FOLDER_NAME = '__INBOX__'

    # Имя-метка для автопапки 'Я автор', хранящееся в базе данных.
    OWNER_AUTOFOLDER_NAME = '__OWNER__'

    # Имя-метка для автопапки 'Я наблюдатель', хранящееся в базе данных.
    WATCHER_AUTOFOLDER_NAME = '__WATCHER__'

    # Имя-метка для автопапки 'Отредактированные мной', хранящееся в базе данных.
    LAST_EDIT_AUTOFOLDER_NAME = '__LAST_EDIT__'

    # список зарезервированных имен папок, которые не могут быть использованы в качестве имен пользовательских папок.
    RESERVED_FOLDER_NAMES_LIST = [
        FAVORITES_FOLDER_NAME,
        OWNER_AUTOFOLDER_NAME,
        WATCHER_AUTOFOLDER_NAME,
        LAST_EDIT_AUTOFOLDER_NAME,
    ]

    AUTOFOLDERS = [
        OWNER_AUTOFOLDER_NAME,
        WATCHER_AUTOFOLDER_NAME,
        LAST_EDIT_AUTOFOLDER_NAME,
    ]

    # Тип папки, соответствующий папке Избранное
    FOLDER_TYPE_FAVORITES = 'F'

    # Тип папки, соответствующий пользовательской папке
    FOLDER_TYPE_CUSTOM = 'C'

    # Тип папки, соответствующий автопапке
    FOLDER_TYPE_AUTO = 'A'

    # Имя папки. Уникально для одного пользователя.
    name = models.CharField(max_length=255)

    # Тип папки.
    type = models.CharField(max_length=1, default=FOLDER_TYPE_CUSTOM)

    # Владелец папки.
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)

    # Организация владельца папки.
    org = models.ForeignKey(Organization, blank=True, null=True, default=None, on_delete=models.CASCADE)

    # Количество закладок в папке.
    favorites_count = models.PositiveIntegerField(default=0)

    # Порядковый номер папки в списке пользователя.
    index_number = models.PositiveIntegerField(default=0)

    def __str__(self):
        return 'name=%s  user=%s' % (self.name, self.user)

    @staticmethod
    def create_default_favorites_folders(user):
        """
        Создать для указанного пользователя обязательную дефолтную папку Избранное и набор автопапок.
        Папки создаются только в том случае, если таких еще у данного пользователя нет.
        Метод актуален для нового пользователя.
        """
        Folder.create_favorites_folder(user)
        Folder.create_autofolders(user)

    @staticmethod
    def create_favorites_folder(user):
        """
        Создать для указанного пользователя обязательную дефолтную папку Избранное, если ее еще у него нет.
        Метод актуален для нового пользователя.
        """
        Folder._create_folder(Folder.FAVORITES_FOLDER_NAME, 0, user, Folder.FOLDER_TYPE_FAVORITES)

    @staticmethod
    def create_autofolders(user):
        """
        Создать для указанного пользователя автопапки, если их еще у него нет.
        Метод актуален для нового пользователя.
        """
        Folder._create_folder(Folder.OWNER_AUTOFOLDER_NAME, 10000, user, Folder.FOLDER_TYPE_AUTO)
        Folder._create_folder(Folder.WATCHER_AUTOFOLDER_NAME, 10001, user, Folder.FOLDER_TYPE_AUTO)
        Folder._create_folder(Folder.LAST_EDIT_AUTOFOLDER_NAME, 10002, user, Folder.FOLDER_TYPE_AUTO)

    @staticmethod
    def _create_folder(name, index, user, type):
        """
        Создать папку для указанного пользователя, если ее еще у него нет.
        Метод применяется строго для создания дефолтных папок (Избранное и автопапки)
        """
        if not Folder.objects.filter(user=user, name=name, org=get_org()).exists():
            new_folder = Folder(name=name, favorites_count=0, user=user, index_number=index, type=type, org=get_org())
            new_folder.save()

    class Meta:
        app_label = 'favorites_v2'
        unique_together = ('user', 'name', 'org')
