import logging
from collections import defaultdict
from typing import List, NamedTuple

from django.db import transaction, connection
from django.conf import settings
from django.db.models import F
from django.contrib.auth import get_user_model

from wiki.favorites_v2.models import AutoBookmark, Folder
from wiki.org import org_ctx, get_org, get_org_or_none, org_user, get_user_orgs
from wiki.pages.models import Page
from wiki.pages.utils.resurrect import restore_deleted_slug
from wiki.utils.models import get_chunked
from wiki.utils.tasks.base import LockedCallableTask

import waffle
from wiki.api_core.waffle_switches import USE_PAGE_FOR_AUTOBOOKMARK

log = logging.getLogger(__name__)


class PageInfoToAutoBookMark(NamedTuple):
    """Прокладка между QuerySet и моделью AutoBookmark"""

    id: int
    title: str
    supertag: str
    last_author__username: str
    modified_at: str


class UpdateAutofoldersTask(LockedCallableTask):

    """
    Обновить закладки в автопапках 'Я автор' и 'Я наблюдатель' для всех пользователей.
    """

    name = 'wiki.update_autofolders'
    logger = logging.getLogger(name)
    time_limit = 60 * 60 * 3  # 7200 сек

    def run(self, *args, **kwargs):
        update_autofolders()


class UpdateOwnerAutofolderTask(LockedCallableTask):

    """
    Обновить закладки в автопапке 'Я автор' для указанного пользователя.
    """

    name = 'wiki.update_owner_autofolder'
    logger = logging.getLogger(name)
    time_limit = 60 * 10  # 600 сек
    lock_name_tpl = 'update_owner_autofolder_{user_id}_{org_id}'

    def run(self, user_id, org_id, *args, **kwargs):
        org = get_org_or_none(org_id)
        with org_ctx(org):
            update_owner_autofolder(user_id)


class UpdateWatcherAutofolderTask(LockedCallableTask):

    """
    Обновить закладки в автопапке 'Я наблюдатель' для указанного пользователя.
    """

    name = 'wiki.update_watcher_autofolder'
    logger = logging.getLogger(name)
    time_limit = 60 * 10  # 600 сек
    lock_name_tpl = 'update_watcher_autofolder_{user_id}_{org_id}'

    def run(self, user_id, username, org_id, *args, **kwargs):
        org = get_org_or_none(org_id)
        with org_ctx(org):
            update_watcher_autofolder(user_id, username)


class UpdateLastEditAutofolderTask(LockedCallableTask):

    """
    Пополнить новой закладкой автопапку 'Отредактированные мной' для указанного пользователя.
    Закладка создается со ссылкой на указанную страницу.
    Если автопапка уже содержит максимально допустимое число закладок - 100,
    то закладка на самую старую по дате изменения страницы удаляется.
    """

    name = 'wiki.update_last_edit_autofolder'
    logger = logging.getLogger(name)
    time_limit = 60 * 10  # 600 сек
    lock_name_tpl = 'update_last_edit_autofolder_{user_id}_{page_id}'

    def run(self, user_id, page_id, *args, **kwargs):
        append_or_update_bookmark(user_id, page_id, Folder.LAST_EDIT_AUTOFOLDER_NAME, 100)


def update_autofolders():
    """
    Обновить все закладки в автопапках 'Я автор' и 'Я наблюдатель' для всех пользователей.
    """
    USER_CHUNK_SIZE = 5000
    users_qs = get_user_model().objects.filter(is_active=True, staff__is_robot=False)

    for user_chunk in get_chunked(users_qs, USER_CHUNK_SIZE):
        for user in user_chunk:
            for org in get_user_orgs(user, assert_empty_org=False):
                with org_ctx(org):
                    update_owner_autofolder(user.id)
                    update_watcher_autofolder(user.id, user.username)

    clean_last_edit_autofolder()


def update_owner_autofolder(user_id: int):
    """
    Обновить закладки в автопапке 'Я автор' для указанного пользователя.
    """
    update_autofolder(
        folder=_get_autofolder(user_id, Folder.OWNER_AUTOFOLDER_NAME),
        pages_info=Page.active.filter(authors__id=user_id, org=get_org())
        .order_by('-modified_at')
        .values_list(*PageInfoToAutoBookMark._fields, named=True),
    )


def update_watcher_autofolder(user_id: int, username: str):
    """
    Обновить все закладки в автопапке 'Я наблюдатель' для указанного пользователя.
    """
    update_autofolder(
        folder=_get_autofolder(user_id, Folder.WATCHER_AUTOFOLDER_NAME),
        pages_info=Page.active.filter(pagewatch__user=username, org=get_org())
        .order_by('-modified_at')[:500]
        .values_list(*PageInfoToAutoBookMark._fields, named=True),
    )


@transaction.atomic
def clean_last_edit_autofolder():
    """
    Очистить папку 'Отредактированные мной' от закладок, ведущих на удаленные страницы.
    """
    if waffle.switch_is_active(USE_PAGE_FOR_AUTOBOOKMARK):
        # уменьшим количество закладок в папках LAST_EDIT_AUTOFOLDER_NAME пользователей
        with connection.cursor() as cursor:
            cursor.execute(
                f"""
            WITH folder_and_count_pages AS(
                SELECT
                    autobookmark.folder_id,
                    COUNT(autobookmark.folder_id) AS count_delete
                FROM
                    favorites_v2_autobookmark AS autobookmark
                INNER JOIN
                    favorites_v2_folder AS folder ON (autobookmark.folder_id = folder.id)
                INNER JOIN
                    pages_page AS page ON (autobookmark.page_id = page.id)
                WHERE
                    (folder.name = '{Folder.LAST_EDIT_AUTOFOLDER_NAME}' AND page.status = 0)
                GROUP BY
                    autobookmark.folder_id
                )

            UPDATE
                favorites_v2_folder AS folder
            SET
                favorites_count = favorites_count - folder_and_count_pages.count_delete
            FROM
                folder_and_count_pages
            WHERE
                folder_and_count_pages.folder_id = folder.id
            """
            )

        # удалим все закладки на неактивные страницы
        AutoBookmark.objects.filter(folder__name=Folder.LAST_EDIT_AUTOFOLDER_NAME, page__status=0).delete()
    else:  # legacy
        deleted_pages = Page.objects.filter(status=0).values_list('supertag', flat=True)

        # восстановить оригинальные значения супертэгов
        deleted_pages = (restore_deleted_slug(supertag) for supertag in deleted_pages)

        bookmarks_to_delete = AutoBookmark.objects.filter(
            folder__name=Folder.LAST_EDIT_AUTOFOLDER_NAME, supertag__in=deleted_pages
        ).values_list('folder_id', 'id')

        folder_ids = defaultdict(list)
        bookmark_ids = []
        for folder_id, bookmark_id in bookmarks_to_delete:
            folder_ids[folder_id].append(bookmark_id)
            bookmark_ids.append(bookmark_id)

        # удалим все закладки на неактивные страницы
        AutoBookmark.objects.filter(id__in=bookmark_ids).delete()

        # пересчитаем число закладок в папках, где было удаление
        for folder_id in folder_ids:
            Folder.objects.filter(id=folder_id).update(
                favorites_count=F('favorites_count') - len(folder_ids[folder_id])
            )


def append_or_update_bookmark(user_id, page_id, folder_name, max_bookmarks_count):
    """
    Добавить новую закладку в указанную автопапку для указанного пользователя или обновить page_modified_at и title
    атрибуты существующей закладки.

    @param user_id: id пользователя, которому добавляем закладку
    @param page_id: id страницы, на которую создается ссылка в закладке
    @param folder_name: имя автопапки, в которой создается закладка
    @param max_bookmarks_count: максимально допустимое количество закладок в указанной автопапке
    """
    # этот запрос вне транзакции, потому что они долгие и вызывают WIKI-7280, например
    pages = Page.active.filter(id=page_id).select_related('last_author')

    if len(pages) == 0:
        log.warn('Cannot get Page:id={0}'.format(page_id))
        return

    page = pages[0]

    with org_ctx(page.org), transaction.atomic():
        folder = _get_autofolder(user_id, folder_name)

        rows = AutoBookmark.objects.filter(folder_id=folder.id, supertag=page.supertag).update(
            page_modified_at=page.modified_at, title=page.title
        )

        if rows == 0:  # добавляем новую закладку, так как существующей не найдено
            bookmark = _create_autobookmark(
                folder=folder,
                page_info=PageInfoToAutoBookMark(
                    id=page.id,
                    title=page.title,
                    supertag=page.supertag,
                    last_author__username=page.last_author.username if page.last_author else '',
                    modified_at=page.modified_at,
                ),
            )
            _write_autobookmarks([bookmark])
            folder.favorites_count += 1
            folder.save()
            if folder.favorites_count > max_bookmarks_count:
                _delete_last_bookmark(folder)


@transaction.atomic
def update_autofolder(folder: Folder, pages_info: List[PageInfoToAutoBookMark]):
    if folder.favorites_count != 0:
        AutoBookmark.objects.filter(folder_id=folder.id).delete()

    bookmarks = [_create_autobookmark(folder, page_info) for page_info in pages_info]
    _write_autobookmarks(bookmarks)

    Folder.objects.filter(id=folder.id).update(favorites_count=len(bookmarks))


def _get_autofolder(user_id: int, name: str) -> Folder:
    try:
        return Folder.objects.get(user_id=user_id, name=name, org=get_org())
    except Folder.DoesNotExist:
        Folder.create_autofolders(org_user().get(id=user_id))
        return Folder.objects.get(user_id=user_id, name=name, org=get_org())


def _delete_last_bookmark(folder: Folder):
    AutoBookmark.objects.filter(folder_id=folder.id).order_by('page_modified_at')[0].delete()
    folder.favorites_count -= 1
    folder.save()


def _create_autobookmark(folder: Folder, page_info: PageInfoToAutoBookMark) -> AutoBookmark:
    """
    Вернуть объект автозакладки не сохраняя его в базу.
    """
    return AutoBookmark(
        folder=folder,
        title=page_info.title,
        url='%s://%s/%s' % (settings.WIKI_PROTOCOL, settings.NGINX_HOST, page_info.supertag),
        supertag=page_info.supertag,
        page_id=page_info.id,
        page_last_editor=page_info.last_author__username or '',
        page_modified_at=page_info.modified_at,
    )


def _write_autobookmarks(bookmarks: List[AutoBookmark]):
    """
    Записать закладки в базу.
    """
    if bookmarks:
        AutoBookmark.objects.bulk_create(bookmarks)
