from celery.utils.log import get_task_logger
from collections import defaultdict

from wiki.favorites_v2.models import AutoBookmark, Bookmark, Folder
from wiki.org import org_ctx, get_org, get_org_or_none
from wiki.pages.models import Page
from wiki.utils.models import get_chunked
from wiki.utils.tasks.base import LockedCallableTask


logger = get_task_logger(__name__)


class UpdateBookmarksTask(LockedCallableTask):

    """
    Обновить в пользовательских папках вики-специфичные атрибуты закладок: автор и дата последних изменений.
    """

    name = 'wiki.update_bookmarks'
    logger = get_task_logger(__name__)
    time_limit = 60 * 60  # 3600 сек

    def run(self, *args, **kwargs):
        update_bookmarks(Bookmark.objects.filter(supertag__isnull=False).select_related('folder__org'), Bookmark)


class UpdateAutoBookmarksTask(LockedCallableTask):

    """
    Обновить в автопапках вики-специфичные атрибуты закладок: автор и дата последних изменений.
    """

    name = 'wiki.update_autobookmarks'
    logger = get_task_logger(__name__)
    time_limit = 60 * 60 * 2  # 72000 сек

    def run(self, *args, **kwargs):
        update_bookmarks(
            AutoBookmark.objects.exclude(folder__name=Folder.LAST_EDIT_AUTOFOLDER_NAME).select_related('folder__org'),
            AutoBookmark,
        )


class UpdateBookmarksBySupertagTask(LockedCallableTask):

    """
    Обновить во всех папках вики-специфичные атрибуты закладок: автор и дата последних изменений.
    Обновление применяется только к закладкам, которые ссылаются на вики страницу с указанным супертегом.
    """

    name = 'wiki.update_bookmarks_by_supertag'
    logger = get_task_logger(__name__)
    time_limit = 60 * 60  # 3600 сек
    lock_name_tpl = 'update_bookmarks_by_supertag_{supertag}_{org_id}'

    def run(self, supertag, org_id, *args, **kwargs):
        logger.info(f'update bookmarks: {supertag}-{org_id}')

        org = get_org_or_none(org_id)
        with org_ctx(org):
            update_bookmarks(
                Bookmark.objects.filter(supertag=supertag, folder__org=get_org()).select_related('folder__org'),
                Bookmark,
            )
            update_bookmarks(
                AutoBookmark.objects.exclude(folder__name=Folder.LAST_EDIT_AUTOFOLDER_NAME)
                .filter(supertag=supertag, folder__org=get_org())
                .select_related('folder__org'),
                AutoBookmark,
            )


def update_bookmarks(bookmarks, bookmark_model):
    BOOKMARK_CHUNK_SIZE = 100
    chunk_number = 0
    for bookmark_chunk in get_chunked(bookmarks, BOOKMARK_CHUNK_SIZE):
        chunk_number += 1

        org_to_bookmarks = defaultdict(list)
        for bookmark in bookmark_chunk:
            org_to_bookmarks[bookmark.folder.org].append(bookmark)

        for org in org_to_bookmarks.keys():
            with org_ctx(org):
                org_bookmarks = org_to_bookmarks[org]

                supertag_ids = defaultdict(list)
                for bookmark in org_bookmarks:
                    supertag_ids[bookmark.supertag].append(bookmark.id)

                supertags = list(supertag_ids.keys())

                if issubclass(bookmark_model, AutoBookmark):
                    # в автопапках обновляем также и 'title'
                    pages = Page.objects.filter(supertag__in=supertags, org=get_org()).values_list(
                        'supertag', 'modified_at', 'last_author__username', 'title'
                    )

                    for supertag, modified_at, last_author, title in pages:
                        bookmark_model.objects.filter(id__in=supertag_ids[supertag]).update(
                            page_modified_at=modified_at, page_last_editor=last_author or '', title=title
                        )
                else:
                    # в пользовательских папках 'title' не обновляем
                    pages = Page.objects.filter(supertag__in=supertags, org=get_org()).values_list(
                        'supertag', 'modified_at', 'last_author__username'
                    )

                    for supertag, modified_at, last_author in pages:
                        bookmark_model.objects.filter(id__in=supertag_ids[supertag]).update(
                            page_modified_at=modified_at, page_last_editor=last_author or ''
                        )
