import logging

from django.contrib.auth import get_user_model

from wiki.favorites_v2.tasks.update_autofolders import (
    UpdateLastEditAutofolderTask,
    UpdateOwnerAutofolderTask,
    UpdateWatcherAutofolderTask,
)
from wiki.favorites_v2.tasks.update_bookmarks import UpdateBookmarksBySupertagTask
from wiki.org import get_org_id, org_ctx
from wiki.pages.models import Page
from wiki.users.logic import uses_new_favorites
from wiki.utils.tasks.base import LockedCallableTask

log = logging.getLogger(__name__)

# WIKI-10487
# Задача, объединяющая в себе запуск подзадач по обновлению закладок
# после редактирования страницы. Нужна, чтобы не ставить несколько
# отдельных задач, для снижения нагрузки на Celery брокера.


class UpdateAutofoldersAndBookmarksAfterPageEditionTask(LockedCallableTask):
    name = 'wiki.update_autofolders_and_bookmarks_after_page_edition'
    logger = logging.getLogger(name)
    time_limit = 60 * 60 * 4  # 4 часа
    lock_name_tpl = 'update_autofolders_and_bookmarks_after_page_edition_{page_id}'

    def run(self, page_id, last_author_id, was_created, *args, **kwargs):
        page = Page.objects.get(id=page_id)
        last_author = get_user_model().objects.get(id=last_author_id)

        with org_ctx(page.org):
            if not last_author.is_robot() and not uses_new_favorites(last_author):
                # обновляем закладки в автопапке 'Отредактированные мной'
                UpdateLastEditAutofolderTask()(user_id=page.last_author.id, page_id=page.id)

                # обновляем закладки в автопапке 'Я наблюдатель'
                UpdateWatcherAutofolderTask()(
                    user_id=page.last_author.id, username=page.last_author.username, org_id=get_org_id()
                )

                # обновляем закладки в автопапке 'Я автор'
                UpdateOwnerAutofolderTask()(user_id=page.last_author.id, org_id=get_org_id())

            if not was_created:
                # уже существовала раньше, поэтому возможно кто-то на нее подписан.
                UpdateBookmarksBySupertagTask()(supertag=page.supertag, org_id=get_org_id())
