from django.db import transaction
from django.shortcuts import get_object_or_404
from pydantic import PositiveInt

from ninja import Query
from wiki.api_v2.collections import Collection, PaginationQuery, CollectionFactory

from wiki.api_v2.di import di
from wiki.api_v2.schemas import DELETED, DELETED_RESPONSE
from wiki.featured_pages.models import LinkGroupFullSchema, LinkGroup, LinkGroupCreateSchema, LinkGroupUpdateSchema
from wiki.sync.connect.base_organization import BaseOrganization

from wiki.api_v2.router_factory import make_support_api_router


@di
def get_link_groups_view(
    request, organization: BaseOrganization, pagination: PaginationQuery = Query(...)
) -> Collection[LinkGroupFullSchema]:
    # Фича пока только для интранета, по org_id никакой фильтрации нет

    return CollectionFactory.build(
        qs=LinkGroup.objects.filter(org=organization.as_django_model()),
        serializer=LinkGroupFullSchema.from_orm,
        pagination=pagination,
    )


@di
def get_link_group_details_view(request, organization: BaseOrganization, idx: PositiveInt) -> LinkGroupFullSchema:
    # Фича пока только для интранета, по org_id никакой фильтрации нет
    obj = get_object_or_404(LinkGroup, pk=idx, org=organization.as_django_model())
    return LinkGroupFullSchema.from_orm(obj)


@di
@transaction.atomic
def create_link_group_view(request, organization: BaseOrganization, data: LinkGroupCreateSchema) -> LinkGroupFullSchema:
    p = LinkGroup(title=data.title, rank=data.rank, org=organization.as_django_model())
    p.set_visibility(data.visibility)
    p.save()
    # we need pk to make many-to-many
    p.set_links([q.to_link_schema(organization) for q in data.links])

    return LinkGroupFullSchema.from_orm(p)


@di
@transaction.atomic
def patch_link_group_view(
    request, organization: BaseOrganization, data: LinkGroupUpdateSchema, idx: int
) -> LinkGroupFullSchema:
    obj = get_object_or_404(LinkGroup, pk=idx, org=organization.as_django_model())

    if data.title:
        obj.title = data.title

    if data.rank:
        obj.rank = data.rank

    if data.visibility:
        obj.set_visibility(data.visibility)

    if data.links:
        obj.set_links([q.to_link_schema(organization) for q in data.links])

    obj.save()
    return LinkGroupFullSchema.from_orm(obj)


@di
@transaction.atomic
def delete_link_group_view(request, organization: BaseOrganization, idx: int):
    obj = get_object_or_404(LinkGroup, pk=idx, org=organization.as_django_model())
    obj.delete()

    return DELETED


featured_pages_admin_api = make_support_api_router(tags=['support_api/featured_pages'])

featured_pages_admin_api.get('/groups', exclude_unset=True, response=Collection[LinkGroupFullSchema])(
    get_link_groups_view
)
featured_pages_admin_api.get('/groups/{idx}/', exclude_unset=True, response=LinkGroupFullSchema)(
    get_link_group_details_view
)
featured_pages_admin_api.post('/groups', exclude_unset=True)(create_link_group_view)
featured_pages_admin_api.post('/groups/{idx}/', exclude_unset=True)(patch_link_group_view)
featured_pages_admin_api.delete('/groups/{idx}/', response=DELETED_RESPONSE)(delete_link_group_view)
