import re

from django.conf import settings

from wiki.api_v2.exceptions import BadRequest
from wiki.featured_pages.user_api.views import UserLazyVisibilityProperties, get_user_visibility
from wiki.pages.models import Page

HOMEPAGE_SPA = 'homepage/spa'

IF_RE = r'^#if'
COMMENT = r'^//'
REMOVE_CMT = r'(//(.*))$'
CONDITIONS = r'(group|city|country|office)=(.*?)\s*$'
ENDIF_RE = r'^#endif'

SCAN = 0
IGNORE = 1
INCLUDE = 2


# https://staff-api.yandex-team.ru/v3/group?_pretty=1&url=yandex_rkub_taxi_dep22593


def check_condition(line, u: UserLazyVisibilityProperties):
    line = re.sub(REMOVE_CMT, '', line)
    conditions = re.findall(CONDITIONS, line)

    if len(conditions) == 0:
        return False

    for condition, values in conditions:
        if ',' in values:
            values = set([int(v.strip()) for v in values.split(',')])
        else:
            values = {int(values)}

        if condition == 'group':
            if values.isdisjoint(set(u.groups)):
                return False
        if condition == 'city':
            if u.geo.city not in values:
                return False
        if condition == 'office':
            if u.geo.office not in values:
                return False
        if condition == 'country':
            if u.geo.country not in values:
                return False

    return True


def preprocess_markup(raw, u: UserLazyVisibilityProperties):
    return '\n'.join(_preprocess_markup(raw, u))


def _preprocess_markup(raw, u):
    depth = 0
    culling_depth = -1
    sm = INCLUDE
    for line in raw.split('\n'):
        if sm == INCLUDE:
            if re.match(IF_RE, line):
                depth += 1
                try:
                    should_include = check_condition(line, u)
                except Exception as e:
                    yield f'!!#if parse error - {str(e)}!!'
                    should_include = False
                if not should_include:
                    sm = IGNORE
                    culling_depth = depth - 1
            elif re.match(ENDIF_RE, line):
                depth -= 1
                if depth < 0:
                    yield '!!extra #endif!!'
                    return
            elif re.match(COMMENT, line):
                pass
            else:
                yield line
        elif sm == IGNORE:
            if re.match(IF_RE, line):
                depth += 1
            elif re.match(ENDIF_RE, line):
                depth -= 1
                if depth == culling_depth:
                    sm = INCLUDE
    if depth != 0:
        yield '!!#if without #endif!!'

    pass


def get_homepage_spa(user, org) -> str:
    if not settings.IS_INTRANET:
        raise BadRequest('Feature only for intranet')

    try:
        page = Page.objects.get(supertag=HOMEPAGE_SPA)
        return preprocess_markup(page.body, get_user_visibility(user, 'ru'))
    except Page.DoesNotExist:
        return '!!no homepage/spa!!'
