import logging
from enum import Enum
from typing import List, Optional, Union, Type, TypeVar

from django.contrib.postgres.fields import JSONField
from django.db import models
from ninja import Schema
from pydantic import BaseModel, HttpUrl, ValidationError, Field

from wiki.api_v2.public.pages.schemas import PageSchema
from wiki.featured_pages.user_api.schemas import LinkSchema, LinkType

from wiki.api_v2.schemas import AllOptional
from wiki.featured_pages.user_api.schemas import LinkEditSchema
from wiki.pages.models import Page
from wiki.sync.connect.models import Organization

logger = logging.getLogger(__name__)


class VisibilityOption(str, Enum):
    VISIBLE = 'visible'
    HIDDEN = 'hidden'


class Affilation(str, Enum):
    STAFF = 'staff'
    OUTSTAFF = 'outstaff'


T = TypeVar('T')


def _try_parse(type_obj: Type[T], obj: Union[str, dict, JSONField], default: T) -> T:
    try:
        if isinstance(obj, str):
            return type_obj.parse_raw(obj)
        return type_obj.parse_obj(obj)
    except ValidationError as err:
        logger.error(f"Can't parse object with the schema: obj={obj}, errors={err.errors()}")
        return default


class GeoFilter(BaseModel):
    city: Optional[int]
    country: Optional[int]
    office: Optional[int]


class VisibilityFilter(BaseModel):
    groups: Optional[List[int]]
    language: Optional[str]
    affilation: Optional[Affilation]
    geo: Optional[GeoFilter]
    is_hidden: bool = False


class DbLinkSchema(BaseModel):
    title: str
    url: HttpUrl
    rank: int = 0


class DbLinksJsonSchema(BaseModel):
    links: List[DbLinkSchema] = Field(default_factory=list)


class LinkGroupToPage(models.Model):
    objects: models.Manager

    page = models.ForeignKey(Page, on_delete=models.CASCADE, related_name='+')
    link_group = models.ForeignKey('LinkGroup', on_delete=models.CASCADE, related_name='+')
    rank = models.IntegerField(default=0)


class LinkGroup(models.Model):
    """
    Группа рекомендуемых страниц в пользовательском меню
    """

    org = models.ForeignKey(Organization, blank=True, null=True, default=None, on_delete=models.CASCADE)

    id: models.IntegerField
    objects: models.Manager

    # Название группы
    title = models.CharField(max_length=255)

    # Для сортировки относительно других групп
    rank = models.IntegerField(default=0)

    # Список страниц в группе
    pages = models.ManyToManyField(Page, through=LinkGroupToPage, blank=True, related_name='+')

    # Список внешних ссылок
    links = JSONField(default=dict)

    # Свойства видимости
    visibility = JSONField(default=dict)

    def __str__(self):
        return f'id={self.id}, title={self.title}, rank={self.rank}'

    def set_visibility(self, mdl: VisibilityFilter):
        self.visibility = mdl.dict()

    def set_external_links(self, links: List[DbLinkSchema]):
        self.links = DbLinksJsonSchema(links=links).dict()

    def set_links(self, links: List[LinkSchema]):
        ext_links = []
        pages = []
        for l in links:
            l: LinkSchema
            if l.type == LinkType.LINK:
                ext_links.append(DbLinkSchema(title=l.title, url=l.external_url, rank=l.rank))
            if l.type == LinkType.PAGE:
                pages.append(LinkGroupToPage(page_id=l.page.id, link_group=self, rank=l.rank))

        self.set_external_links(ext_links)
        LinkGroupToPage.objects.filter(link_group=self).delete()
        LinkGroupToPage.objects.bulk_create(pages)

    def get_visibility(self) -> VisibilityFilter:
        return _try_parse(VisibilityFilter, self.visibility, VisibilityFilter(is_hidden=True))

    def get_external_links(self) -> List[DbLinkSchema]:
        return _try_parse(DbLinksJsonSchema, self.links, DbLinksJsonSchema()).links


class CachedPageSchema(Schema):
    id: int
    title: str
    slug: str
    rank: int


class LinkGroupFullSchema(BaseModel):
    id: int
    title: str
    rank: int
    links: List[LinkSchema] = Field(default_factory=list)
    visibility: VisibilityFilter

    class Config:
        orm_mode = True

    @classmethod
    def from_orm(cls, orm_model: LinkGroup):
        all_links = [
            LinkSchema(type=LinkType.LINK, external_url=link.url, title=link.title, rank=link.rank)
            for link in orm_model.get_external_links()
        ] + [
            LinkSchema(
                type=LinkType.PAGE, page=PageSchema(id=q.page.id, slug=q.page.slug), title=q.page.title, rank=q.rank
            )
            for q in LinkGroupToPage.objects.filter(link_group_id=orm_model.id).prefetch_related('page').all()
        ]
        all_links.sort(key=lambda link: (link.rank, link.title))

        return cls(
            id=orm_model.id,
            title=orm_model.title,
            rank=orm_model.rank,
            links=all_links,
            visibility=orm_model.get_visibility(),
        )


class LinkGroupCreateSchema(BaseModel):
    title: str
    rank: int
    visibility: VisibilityFilter = Field(default_factory=lambda: VisibilityFilter(is_hidden=False))
    links: List[LinkEditSchema] = Field(default_factory=list)


class LinkGroupUpdateSchema(LinkGroupCreateSchema, metaclass=AllOptional):
    pass
