from enum import Enum
from typing import List, Optional

from ninja import Schema
from pydantic import Field, HttpUrl, root_validator

from wiki.api_v2.public.pages.page_identity import PageIdentity, resolve_page_identity
from wiki.api_v2.public.pages.schemas import PageSchema
from wiki.sync.connect.base_organization import BaseOrganization


class LinkType(str, Enum):
    PAGE = 'page'
    LINK = 'link'


class LinkSchema(Schema):
    type: LinkType
    title: Optional[str]
    rank: int = Field(default=0)

    page: Optional[PageSchema]
    external_url: Optional[HttpUrl]


class LinkEditSchema(Schema):
    title: Optional[str]
    rank: int = Field(default=0)

    page: Optional[PageIdentity]
    external_url: Optional[HttpUrl]

    def to_link_schema(self, organization: BaseOrganization):
        page = self.page
        if page:
            page = PageSchema.from_orm(resolve_page_identity(organization, page))
        return LinkSchema(
            type=LinkType.LINK if self.external_url else LinkType.PAGE,
            title=self.title,
            rank=self.rank,
            page=page,
            external_url=self.external_url,
        )

    @root_validator
    def root_validate(cls, data):
        has_page = bool(data.get('page', None))
        has_external_url = bool(data.get('external_url', None))

        if has_external_url and not data.get('title'):
            raise ValueError('Must provide title for external links')

        if not (has_page ^ has_external_url):
            raise ValueError('Must provide external_url or page field')

        return data


class LinkGroupSchema(Schema):
    id: int
    title: str
    links: List[LinkSchema] = Field(description='List of links and pages sorted by title')

    def sort_links(self):
        self.links.sort(key=lambda link: (link.rank, link.title))
        return self


class FeaturedPagesResponseSchema(Schema):
    groups: List[LinkGroupSchema]
