from typing import Dict, List

from django.conf import settings
from django.contrib.auth import get_user_model

from wiki.api_v2.di import di, legacy_org_ctx
from wiki.featured_pages.user_api.schemas import FeaturedPagesResponseSchema, LinkGroupSchema, LinkType
from wiki.featured_pages.models import (
    LinkGroup,
    LinkGroupFullSchema,
    VisibilityFilter,
    Affilation,
    GeoFilter,
)
from wiki.pages.access import get_bulk_access_status, ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED
from wiki.pages.access.groups import user_django_groups, user_group_ids
from wiki.sync.connect.base_organization import BaseOrganization

User = get_user_model()


class UserLazyVisibilityProperties:  # implements VisibilityFilter
    def __init__(self, user, language):
        self.language = language
        self.user = user

    @property
    def affilation(self) -> Affilation:
        if settings.IS_INTRANET:
            is_staff = any(
                group.name == settings.IDM_ROLE_EMPLOYEE_GROUP_NAME for group in user_django_groups(self.user)
            )
            return Affilation.STAFF if is_staff else Affilation.OUTSTAFF
        return Affilation.STAFF

    @property
    def groups(self):
        return user_group_ids(self.user.staff)

    @property
    def geo(self) -> GeoFilter:
        if settings.IS_INTRANET and self.user.staff and hasattr(self.user.staff, 'office'):
            office = self.user.staff.office
            city = office and office.city
            country = city and city.country
            return GeoFilter(office=office and office.id, city=city and city.id, country=country and country.id)

        return GeoFilter()


def get_user_visibility(user: User, language: str) -> UserLazyVisibilityProperties:
    return UserLazyVisibilityProperties(user, language)


def get_link_groups_from_cache(organization: BaseOrganization) -> List[LinkGroupFullSchema]:  # TODO add cache
    link_groups = LinkGroup.objects.filter(org=organization.as_django_model())
    schemas = [LinkGroupFullSchema.from_orm(featured_page) for featured_page in link_groups]
    schemas.sort(key=lambda x: x.rank)
    return schemas


def is_accessible_to_user(visibility_filter: VisibilityFilter, user: UserLazyVisibilityProperties) -> bool:
    if visibility_filter.is_hidden:
        return False

    if visibility_filter.affilation and visibility_filter.affilation != user.affilation:
        return False

    if visibility_filter.groups and not set(visibility_filter.groups) & set(user.groups):
        return False

    if visibility_filter.language and visibility_filter.language != user.language:
        return False

    if visibility_filter.geo:
        if visibility_filter.geo.office and visibility_filter.geo.office != user.geo.office:
            return False

        if visibility_filter.geo.city and visibility_filter.geo.city != user.geo.city:
            return False

        if visibility_filter.geo.country and visibility_filter.geo.country != user.geo.country:
            return False

    return True


def get_page_accessibility(groups: List[LinkGroupFullSchema], user: User) -> Dict[str, bool]:  # TODO add cache
    slugs = set()
    for group in groups:
        slugs |= {lnk.page.slug for lnk in group.links if lnk.type == LinkType.PAGE}

    access = get_bulk_access_status(tags=slugs, user=user)
    valid_access = {ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED}
    return {slug: access[slug] in valid_access for slug in slugs}


@di
@legacy_org_ctx
def get_featured_pages_view(request, organization: BaseOrganization) -> FeaturedPagesResponseSchema:
    user_visibility = get_user_visibility(user=request.user, language=request.LANGUAGE_CODE)

    groups = get_link_groups_from_cache(organization)
    accessible_groups = [group for group in groups if is_accessible_to_user(group.visibility, user_visibility)]

    accessible_pages = get_page_accessibility(accessible_groups, user=request.user)

    for group in accessible_groups:
        group.links = [lnk for lnk in group.links if lnk.type == LinkType.LINK or accessible_pages[lnk.page.slug]]

    link_groups = [LinkGroupSchema.from_orm(group) for group in accessible_groups if group.links]

    return FeaturedPagesResponseSchema(groups=link_groups)
