"""
Common files operations
"""

import logging
import mimetypes
from datetime import timedelta
from urllib.parse import quote, urlencode

import requests
import waffle
from PIL import Image
from django.conf import settings
from django.db import transaction
from django.http import HttpResponseRedirect, StreamingHttpResponse
from requests import ReadTimeout, ConnectTimeout

from wiki.api_core.waffle_switches import DELIVER_FILES_VIA_SIGNED_URL
from wiki.api_v2.public.upload_sessions.exceptions import StorageUploadError
from wiki.files import logic as files_logic
from wiki.files.models import File, make_storage_id
from wiki.files.models import MDS_STORAGE as STORAGE
from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.pages.access import is_admin
from wiki.pages.models import Page
from wiki.uploads.s3_client import S3_CLIENT, MOCKED_STORAGE_HOST
from wiki.utils import timezone
from wiki.utils.features.get_features import get_wiki_features

logger = logging.getLogger('wiki.files')

FILE_DOWNLOAD_CHUNK_SIZE = 5 * (1 << 20)  # 5Mb


def delete_file(file_id, user):
    try:
        file = File.active.select_related('user', 'page').get(id=file_id)
    except File.DoesNotExist:
        logger.warning('file %s does not exist', file_id)
        return True

    # if not self.page.check_acl(user, 'write') and ((not user) or f.user!=user.name):
    # django user (auth_user)
    if not user.is_authenticated or (user not in file.page.get_authors() and file.user != user and not is_admin(user)):
        return False

    files_logic.delete(file, user=user)

    return True


def upload_file(uploaded_file, page, user, description='', check_access=True):
    """
    user - users.User
    """
    if check_access and not page.has_access(user, 'write'):
        return False

    now = timezone.now()

    f = File(
        page=page,
        user=user,
        name=uploaded_file.name,
        url=File.get_unique_url(page, uploaded_file.name),
        size=uploaded_file.size,
        description=description,
        created_at=now,
        modified_at=now,
    )
    with transaction.atomic():
        if hasattr(uploaded_file, 'storage_id'):
            f.mds_storage_id = uploaded_file.storage_id
            f.save()
        else:
            # пересохраняем, чтобы f получила pk.
            f.save()
            f.mds_storage_id = STORAGE.save(make_storage_id(f.name, f.size), uploaded_file)
            f.save()

        page = Page.objects.get(pk=page.pk)
        page.files += 1
        page.save()

        PageEvent(
            timeout=now + timedelta(minutes=20),
            page=page,
            author=user,
            meta={'filename': f.name, 'url': f.url, 'size': f.size, 'id': f.id},
            event_type=EventTypes.add_file,
            notify=True,
        ).save()
    return f


def detect_image(file_path):
    """
    Tries to detect image in filesystem's file_path.
    Returns tuple with dimensions of image if file is image,
    otherwise returns (0, 0).
    """
    try:
        return Image.open(file_path).size
    except IOError:
        return (0, 0)


def _derive_content_type(filename):
    mime_type, encoding = mimetypes.guess_type(filename)
    if not mime_type:
        return 'application/octet-stream;'
    elif encoding:
        return '; charset='.join([mime_type, encoding])
    else:
        return mime_type


def quote_path(path):
    if isinstance(path, str):
        path = path.encode('utf-8')

    return quote(path)


def file_download_response(file_object, force_download=False):
    if waffle.switch_is_active(DELIVER_FILES_VIA_SIGNED_URL):
        if file_object.is_stored_on_s3:
            return redirect_to_s3_signed_url(file_object, force_download, settings.MDS_SIGNED_URL_EXPIRATION_SEC)
        return redirect_to_mds_signed_url(file_object, force_download, settings.MDS_SIGNED_URL_EXPIRATION_SEC)
    else:
        return response_to_download_file(file_object, force_download)


def redirect_to_mds_signed_url(file_object, force_download=False, expiration_time=60):
    if get_wiki_features().mock_attach_download:
        return HttpResponseRedirect(f'{MOCKED_STORAGE_HOST}{file_object.mds_storage_id}')

    content_type = _derive_content_type(file_object.name)
    storage_url = file_object.mds_storage_id.storage.url(file_object.mds_storage_id.name)

    uri = '%s?%s' % (
        storage_url,
        urlencode(
            {
                'expiration-time': expiration_time,
                'content-type': content_type,
                'filename': file_object.name,
                'disposition': 'inline',
                'redirect': 'yes',
            }
        ),
    )

    # иногда (редко, но стабильно) MDS подлагивает и может отдавать хед секунд по 40.
    # попробуем получить с четким таймаутом.

    attempts = 3
    timeout_dur = 1

    while attempts > 0:
        attempts -= 1

        try:
            mds_redirect_response = requests.head(uri, timeout=timeout_dur)
            location = mds_redirect_response.headers.get('Location')
            if location and mds_redirect_response.status_code == 302:
                return HttpResponseRedirect(location)
            else:
                logger.warning(f'MDS Failed. Status {mds_redirect_response.status_code}')
                break

        except (ConnectTimeout, ReadTimeout) as e:
            logger.info(f'MDS HEAD failed to meet timeout {e}. Attempts left {attempts}')
            timeout_dur = 2 * timeout_dur

    logger.info('MDS Failed to deliver redirect for file %s, using streaming fallback' % file_object.id)
    return response_to_download_file(file_object, force_download=force_download)


def redirect_to_s3_signed_url(file_object, force_download=False, expiration_time=60):
    try:
        response_headers = {
            'content-type': _derive_content_type(file_object.name),
            'content-disposition': '''inline; filename="{0}"; filename*="UTF-8''{1}"'''.format(
                file_object.name,
                quote(file_object.name),
            ),
            'Cache-Control': f'max-age={expiration_time},immutable',
        }
        signed_url = S3_CLIENT.generate_presigned_url(
            key=file_object.s3_storage_key,
            expires_in=expiration_time,
            response_headers=response_headers,
        )
    except StorageUploadError:
        return response_to_download_file(file_object, force_download=force_download)

    return HttpResponseRedirect(signed_url)


def response_to_download_file(file_object, force_download: bool = False):
    """
    Получить HttpResponse для пользователя, который пришел за файлом.
    @type file_object: File
    @type request: HttpRequest
    @return: HttpResponse
    """
    if file_object.is_stored_on_s3:
        response = _response_to_download_s3_file(file_object)
    else:
        response = _response_to_download_mds_file(file_object)

    download = force_download or files_logic.should_download(file_object.name)

    if download:
        response['Content-Disposition'] = '''attachment; filename="{0}"; filename*="UTF-8''{1}"'''.format(
            file_object.name,
            quote(file_object.name),
        )
        response['Content-Transfer-Encoding'] = 'binary'

    # WIKI-12345
    response['X-Content-Type-Options'] = 'nosniff'

    # так надо для Safari, https://st.yandex-team.ru/WIKI-8824 и дальше на stackoverflow
    if not response.has_header('Last-Modified'):
        response['Expires'] = file_object.modified_at.strftime('%a, %d %b %Y %H:%M:%S GMT')

    # max-age - максимальный период времени в сек, в течение которого контент остается свежим
    # immutable указывает, что тело ответа не изменится с течением времени
    response['Cache-Control'] = 'max-age=3600,immutable'

    return response


def _response_to_download_s3_file(file_object):
    data_stream = S3_CLIENT.get_object_body_stream(file_object.s3_storage_key)
    response = StreamingHttpResponse(
        data_stream.iter_chunks(FILE_DOWNLOAD_CHUNK_SIZE), content_type=_derive_content_type(file_object.name)
    )
    return response


def _response_to_download_mds_file(file_object):
    storage_url = file_object.mds_storage_id.storage.url(file_object.mds_storage_id.name)

    if get_wiki_features().mock_attach_download:
        mds_response = requests.Response()
    else:
        # Отдаем файл кусками, чтобы большой файл не забил всю оперативку.
        mds_response = requests.get(storage_url, stream=True)

    # пытаемся выяснить корректный content-type ответа
    response = StreamingHttpResponse(
        mds_response.iter_content(chunk_size=FILE_DOWNLOAD_CHUNK_SIZE),
        content_type=_derive_content_type(file_object.name),
    )

    if mds_response.headers.get('Content-Encoding') == 'gzip':
        del mds_response.headers['Content-Encoding']
    if mds_response.headers.get('Transfer-Encoding') == 'chunked':
        del mds_response.headers['Transfer-Encoding']
    if 'Content-Type' in mds_response.headers:
        del mds_response.headers['Content-Type']

    for header_name, header_value in mds_response.headers.items():
        response[header_name] = header_value

    return response
