import mimetypes

from django.conf import settings
from django.http import HttpResponse

from wiki.files import dao as files_dao
from wiki.files.models import File, MDS_STORAGE as STORAGE
from wiki.pages import dao as pages_dao
from wiki.pages.models import RedirectLoopException
from wiki.utils import xaccel_header
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils.supertag import translit

# те типы файлов, которые через вики разрешено смотреть в браузере
_VIEW_WITH_BROWSER = {
    'image/gif': 'gif',
    'image/jpeg': 'jpg',
    'image/png': 'png',
    'text/html': 'html',
}


def make_key(key):
    """
    Сделать ключ, чтобы было удобно хранить в хранилище.

    @type key: basestring
    """
    return '{instance}:{key}'.format(instance=settings.WIKI_CODE, key=key)


def save_into_storage(key, data):
    """
    Сохранить в хранилище.

    Сохраняем минуя проверку exists у хранилища, т.е. с перезаписью.
    """
    STORAGE.save(make_key(key), data)


def http_response_by_storage_key(key):
    """
    Вернуть HttpResponse чтобы отдать данные из хранилища через nginx.

    Используется X-Accel-Redirect.
    Вы сперва должны положить ваши данные через save_into_storage, чтобы
    эта функция корректно отработала.
    """
    # TODO: требуется рефакторинг в WIKI-7662
    response = HttpResponse(content='', content_type='application/json')
    url = xaccel_header.check_address_in_url(STORAGE.url(make_key(key)))
    response['X-Accel-Redirect'] = xaccel_header.ACCEL_REDIRECT_SLUG + url
    response['X-Accel-Buffering'] = 'no'  # не кэшируем отданные файлы
    return response


def delete(file, user, page=None):
    """
    Правильный способ удалить файл.

    page — страница, к которой прикреплен файл. Имеет смысл передавать,
    если она есть, а в самом файле обращение к page вызовет доп. запрос
    """
    page = page or file.page

    pages_dao.decrement_files_count(page=page)
    files_dao.create_delete_file_event(file=file, user=user, page=page)
    files_dao.delete(file=file)


def restore(file):
    if not file.is_deleted:
        return

    files_dao.restore(file)
    pages_dao.increment_files_count(page=file.page)


def should_download(file_name):
    """
    Всегда скачивать этот файл, никогда не позволять отображать контент в браузере.

    При выборе приоритет остается за скачиванием, вместо отображения в браузере.

    @type file_name: basestring
    @type download_parameter: basestring
    @rtype: bool
    """
    mime_type, encoding = mimetypes.guess_type(file_name)

    if mime_type and 'html' in mime_type and settings.IS_BUSINESS:
        by_type = True
    else:
        by_type = mime_type not in _VIEW_WITH_BROWSER

    return by_type


def file_for_slug_by_filename(filename: str, organization: BaseOrganization, slug: str):
    candidates = organization.get_location_history().filter(slug=slug).order_by('-id')
    if len(candidates) == 0:
        return None

    possible_filenames = {filename, translit(filename)}
    files = File.active.filter(url__in=possible_filenames, page_id__in=[c.page_id for c in candidates])[:50]

    # найти первый подходящий файл из цепочки редиректов
    for candidate in candidates:
        for file in files:
            if file.page_id == candidate.page_id:
                return file

    return None


def file_for_page_by_filename(filename, page):
    """
    Учитывая редиректы, найти файл, прикрепленный к странице.

    @rtype: File
    """
    all_redirects = []
    try:
        for page in page.redirect_chain():
            all_redirects.append(page)
    except RedirectLoopException:
        # использовать те редиректы, которые успели найти
        pass
    possible_filenames = {filename, translit(filename)}
    files = File.active.filter(url__in=possible_filenames, page__in=all_redirects)[:1000]  # 1000 достаточно

    # найти первый подходящий файл из цепочки редиректов
    for page in all_redirects:
        for file in files:
            if file.page == page:
                return file
