import base64
import logging
import mimetypes

from os import path
from random import randint

from django.conf import settings
from django.contrib.postgres.fields import JSONField
from django.db import models

from wiki.cloudsearch.indexable_model_mixin import IndexableModelMixin
from wiki.files.consts import ThumbnailStatus, ThumbnailFieldSchema
from wiki.pages.models import Page
from wiki.pages.models.consts import ACTUALITY_STATUS
from wiki.utils import timezone
from wiki.utils.supertag import translit

logger = logging.getLogger(__name__)


def get_class_by_name(class_path):
    module_name, class_name = class_path.rsplit('.', 1)

    try:
        return getattr(__import__(module_name, {}, {}, [class_name]), class_name)
    except ImportError:
        logger.error('Can not import class by path: %s', class_path)
        raise


MDS_STORAGE = get_class_by_name(settings.STORAGE_CLASS)()


class StoredFileLazy(object):
    """
    Обертка над файлом, грузящая его тело лишь в случае необходимости
    """

    files_storage = MDS_STORAGE

    def __init__(self, storage_id, file_name, size, **kwargs):
        self.storage_id = storage_id
        self.name = file_name
        self.size = size

        self.content_type, _ = mimetypes.guess_type(self.name)
        self._file = None

    @property
    def file(self):
        """
        Вернуть файл из хранилища.
        @rtype: ContentFile
        """
        if self._file is None:
            logger.debug('fetching file "%s" id="%s" from storage', self.name, self.storage_id)
            self._file = self.files_storage.open(self.storage_id)

        return self._file

    def read(self, *args, **kwargs):
        return self.file.read(*args, **kwargs)

    def tell(self, *args, **kwargs):
        return self.file.tell(*args, **kwargs)

    def seek(self, *args, **kwargs):
        return self.file.seek(*args, **kwargs)

    def remove_from_storage(self):
        """"""
        logger.debug('deleting file %s id=%s from storage', self.name, self.storage_id)
        self.files_storage.delete(self.storage_id)

    def __str__(self):
        return self.name


class ActiveFileManager(models.Manager):
    def get_queryset(self):
        qs = super(ActiveFileManager, self).get_queryset()
        return qs.filter(status__gt=0)


def upload_to(instance, filename):
    return True


class File(models.Model, IndexableModelMixin):
    page = models.ForeignKey(Page, on_delete=models.CASCADE)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    name = models.CharField(max_length=255)
    url = models.CharField(max_length=255, db_index=True)
    mds_storage_id = models.FileField(
        max_length=512, storage=MDS_STORAGE, upload_to=upload_to, default='', blank=True, null=True
    )
    s3_storage_key = models.CharField(max_length=512, null=True)
    description = models.CharField(
        max_length=255,
        default='',
        blank=True,
    )
    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True)
    size = models.PositiveIntegerField(default=0)
    image_width = models.PositiveIntegerField(default=0)
    image_height = models.PositiveIntegerField(default=0)
    thumbnail = JSONField(null=True, blank=True)

    status = models.PositiveIntegerField(default=1)

    objects = models.Manager()
    active = ActiveFileManager()

    @classmethod
    def is_image(self, filename):
        filename = translit(filename)
        mimetype, _nothing = mimetypes.guess_type(filename)
        if mimetype:
            return mimetype.startswith('image/')
        return False

    @staticmethod
    def get_unique_url(page, file_name):
        from wiki.pages import dao

        url = translit(file_name)
        i = 1
        try:
            while dao.get_files(page=page, url=url):
                t = path.splitext(url)
                old_sfx = '-%d' % (i - 1)
                pos = t[0].rfind(old_sfx)
                if pos > 0 and pos == len(t[0]) - len(old_sfx):
                    # try to cut previous suffix
                    tt = t[0][: -len(old_sfx)]
                else:
                    tt = t[0]
                url = tt + ('-%d' % i) + t[1]
                i += 1
        except File.DoesNotExist:
            pass
        return url

    def __str__(self):
        return 'File %s on page %s uploaded by %s' % (self.url, self.page.supertag, self.user.username)

    def delete(self, using=None):
        from wiki.files import dao

        dao.delete(self)

    @staticmethod
    def full_path(url, file_url):
        """
        @rtype: unicode
        """
        return '{0}/.files/{1}'.format(url, file_url)

    @staticmethod
    def absolute_url(page_url, file_url):
        """
        Ссылка на скачивание файла.

        @rtype: unicode
        """
        return '{protocol}://{hostname}{path}/'.format(
            protocol=settings.WIKI_PROTOCOL, hostname=settings.NGINX_HOST, path=File.full_path(page_url, file_url)
        )

    @property
    def is_active(self):
        return self.status == 1

    @property
    def is_deleted(self):
        return not self.is_active

    @property
    def storage_url(self):
        if self.s3_storage_key:
            return self.s3_storage_key

        storage_id = self.mds_storage_id
        return storage_id.storage.url(storage_id.name)

    @property
    def is_stored_on_s3(self):
        return bool(self.s3_storage_key)

    @property
    def thumbnail_data(self):
        if self.thumbnail:
            return ThumbnailFieldSchema.deserialize(self.thumbnail)
        return ThumbnailFieldSchema(status=ThumbnailStatus.UNKNOWN)

    @property
    def has_preview(self):
        return self.thumbnail_data.status == ThumbnailStatus.CREATED

    class Meta:
        unique_together = (('page', 'url'),)

    def make_thumbnail_s3_key(self):
        assert self.id is not None, 'File object should have id to create thumbnail storage key'
        created = self.created_at.strftime('%Y-%m-%d')
        return f'{settings.WIKI_CODE}/thumbnails/{created}/{self.id}'

    def get_download_url(self):
        url = self.absolute_url(self.page.url, self.url)
        url = url.rstrip('/')

        return f'{url}?download=1'

    def get_metadata(self):
        return {
            'url': self.get_download_url(),
            'created_at': int(self.created_at.timestamp()),  # TODO utc?
            'modified_at': int(self.modified_at.timestamp()),
            'authors': [{'uid': self.user.get_uid(), 'cloud_uid': self.user.get_cloud_uid()}],
            'is_obsolete': (self.page.actuality_status == ACTUALITY_STATUS.obsolete),  # ACTUALITY_STATUS
            'slug': self.page.supertag,
            'type': 'file',
        }

    def get_document(self):
        return {
            'title': self.name,  # название файла
            'keywords': self.page.get_keywords(),
        }

    def get_search_uuid(self):
        return f'wf:{self.id}'

    def get_acl_subject(self):
        return self.page

    def get_model_org(self):
        return self.page.get_model_org()

    def get_children_files(self):
        return []

    def get_children_pages(self):
        return []


def fileinfo_by_storage_id(storage_id):
    """
    Вернуть пару (имя файла, размер файла).

    @type storage_id: str
    @rtype: (str, int)
    """
    words = storage_id.split(':')
    file_name, file_size = words[2], words[3]
    # декодируем имя файла из base64
    file_name = base64.b64decode(file_name.encode()).decode()
    return file_name, int(file_size)


def make_storage_id(name, size):
    """
    Схема storage_id для файлов.

    base64 не использует в качестве буквы своего алфавита двоеточие ":", поэтому
    можно рассчитывать, что число разделителей у результата этой функции
    всегда фиксированное (http://tools.ietf.org/html/rfc3548.html)
    """
    now = timezone.now()

    # тут не бывает двоеточия по стандарту.
    filename = base64.b64encode(name.encode('utf-8')).decode('utf-8')

    return '{prefix}:file:{name}:{size}:{modified_at}:{random_int}'.format(
        prefix=settings.WIKI_CODE,
        name=filename,
        size=size,
        modified_at=now.strftime('%Y-%m-%d %H:%M:%S:%f'),
        random_int=str(randint(10**6, 10**7 - 1)),
    )
