import logging

from PIL import Image, UnidentifiedImageError
from io import BytesIO
from ylog.context import log_context

from wiki.api_v2.public.upload_sessions.exceptions import StorageUploadError
from wiki.files.models import File
from wiki.files.consts import ThumbnailStatus, ThumbnailFieldSchema
from wiki.uploads.s3_client import S3_CLIENT
from wiki.utils.tasks.base import LockedCallableTask
from wiki.utils.timezone import now

THUMBNAIL_SIZE = 140, 140


class CreateThumbnailTask(LockedCallableTask):
    name = 'wiki.create_image_thumbnail'
    logger = logging.getLogger(__name__)
    time_limit = 120  # 2 минуты
    lock_name_tpl = 'create_image_thumbnail_{file_id}'

    def run(self, file_id, *args, **kwargs):
        with log_context(file_id=file_id):
            try:
                file_obj = File.active.get(pk=file_id)
            except File.DoesNotExist:
                self.logger.warning('File does not exist')
            else:
                self.logger.debug('Create thumbnail for file %s', file_obj.name)
                if file_obj.is_stored_on_s3:
                    file_data = S3_CLIENT.get_object_body_stream(file_obj.s3_storage_key)
                else:
                    file_data = BytesIO(file_obj.mds_storage_id.read())

                thumbnail_data = ThumbnailFieldSchema(
                    status=ThumbnailStatus.CREATED,
                    changed_at=now(),
                )

                try:
                    im = Image.open(file_data)
                    im.thumbnail(THUMBNAIL_SIZE)

                    thumbnail_body = BytesIO()
                    im.save(thumbnail_body, format='PNG')
                    thumbnail_body.seek(0)

                    s3_key = file_obj.make_thumbnail_s3_key()
                    S3_CLIENT.put_object(s3_key, thumbnail_body)
                    thumbnail_data.s3_key = s3_key
                except UnidentifiedImageError:
                    self.logger.error('File is not valid image')
                    thumbnail_data.status = ThumbnailStatus.NOT_AVAILABLE
                except StorageUploadError as e:
                    self.logger.error(f'Failed to save thumbnail, storage error: {e}')
                    thumbnail_data.status = ThumbnailStatus.FAILED

                file_obj.thumbnail = thumbnail_data.serialize()
                file_obj.save()
