from dataclasses import dataclass, field
from enum import Enum
from typing import Any, List, Optional, Dict, Union
from ninja import Schema
from pydantic import Field


class SortDirection(str, Enum):
    ASC = 'asc'
    DESC = 'desc'


class GridColumnTypes(str, Enum):
    STRING = 'string'
    NUMBER = 'number'
    DATE = 'date'
    SELECT = 'select'
    STAFF = 'staff'
    CHECKBOX = 'checkbox'
    TICKET = 'ticket'
    # ----
    TICKET_ASSIGNEE = 'ticket-assignee'
    TICKET_REPORTER = 'ticket-reporter'
    TICKET_STATUS = 'ticket-status'
    TICKET_PRIORITY = 'ticket-priority'
    TICKET_TYPE = 'ticket-type'
    TICKET_SUBJECT = 'ticket-subject'
    TICKET_UPDATED_AT = 'ticket-updatedat'
    TICKET_CREATED_AT = 'ticket-createdat'
    TICKET_ORIGINAL_ESTIMATION = 'ticket-originalestimation'
    TICKET_FIX_VERSIONS = 'ticket-fixversions'
    TICKET_COMPONENTS = 'ticket-components'
    TICKET_STORYPOINTS = 'ticket-storypoints'


class ColumnSchema(Schema):
    type: GridColumnTypes
    id: str = Field(
        description='Автогенерируемый идентификатор колонки. Во избежание двусмысленности, переименован из `name`',
        alias='name',
    )
    title: str = ''

    width: Optional[str] = Field(description='Ширина колонки')
    width_units: Optional[str] = Field(description='Единицы измерения, обычно %')

    # --- специфичные поля для типов
    multiple: Optional[bool] = Field(
        description='Только для типов `staff` и `select`; принимает один или несколько ответов'
    )
    options: Optional[List[str]] = Field(
        description='Только для типов `staff`; Варианты из которых можно выбрать ответ'
    )
    mark_row_if_checked: Optional[bool] = Field(
        alias='markdone',
        description='Только для типа `checkbox`: '
        'Влияет на то как визуально показывается на фронте строчка '
        'грида если в данной колонке стоит птичка: Если '
        '`true` и птичка стоит, вся колонка будет отрисована '
        'блеклым цветом.\n\nБывший `markdone`',
    )
    format: Optional[str] = Field(
        description='Для типа `number`, `date` и `staff`; Определяет то как отрисовать формат'
    )


# access_structure:


@dataclass
class Structure:
    done: Optional[bool]
    fields: List[ColumnSchema] = field(default_factory=list)
    sorting: List[str] = field(default_factory=list)
    title: str = ''


# access_idx:

idx = Dict[str, int]  # row_name -> row order (?)


@dataclass
class Cell:
    raw: Any
    sort: Optional[int] = None
    transformed: Optional[Any] = None
    view: Optional[Any] = None


# data = имя колонки: Cell + поле __key__;


class GridSortSchema(Schema):
    type: SortDirection
    name: str = Field(description='Имя колонки')


class CellSchema(Schema):
    row_id: str
    col_id: str = Field(
        alias='__key__', description='Соответствует `column.id` в `structure`; В apiv1 поле называлось `__key__`'
    )

    raw: Union[Any] = Field(description='Сырое значение;')
    sort: Optional[Any] = Field(description='Вспомогательное поле, будет использоваться при сортировке колонки')
    url: Optional[str] = Field(description='Для типа `ticket` урл на Трекер')
    status: Optional[Any] = Field(description='Для типа `ticket`, удалось ли получить данные')
    transformed: Optional[Any] = Field(description='Для типа `staff` - вернет детали пользователя')


class GridStructureSchema(Schema):
    title: str
    sorting: Optional[List[GridSortSchema]]
    fields: Optional[List[ColumnSchema]]


class GridContentSchema(Schema):
    rows: Optional[List[List[CellSchema]]]
    structure: GridStructureSchema
    version: str

    @classmethod
    def from_orm(cls, obj: Any):
        return GridContentSchema(
            version=obj.get_page_version(),
            structure=GridStructureSchema.parse_obj(obj.structure),
            rows=[[CellSchema.parse_obj(q) for q in row] for row in obj.rows],
        )
