import datetime

from wiki.utils import timezone
from django.db import transaction

import wiki.favorites_v2.logic as favorites_logic

from wiki.grids.models import Grid
from wiki.org import get_org
from wiki.pages.models import PageWatch
from wiki.pages import signals as page_signals
from wiki.notifications.models import PageEvent
from wiki.utils.supertag import tag_to_supertag


def create_grid(tag, title, author_of_changes):
    """
    Создать табличный список.

    @type tag: basestring
    @param tag: обязательно должен быть валидным тегом. Валидации в функции не происходит.
    @type title: basestring
    @type author_of_changes: User
    @rtype: Grid
    """
    now = timezone.now()
    supertag = tag_to_supertag(tag)

    # создание нового грида
    grid = Grid(
        owner=author_of_changes,
        page_type=Grid.TYPES.GRID,
        tag=tag,
        last_author=author_of_changes,
        modified_at=now,
        supertag=supertag,
        org=get_org(),
        with_new_wf=True,
    )
    grid.access_structure['title'] = title
    # это для обратной совместимости со старым фронтом.
    # нужно удалить, когда старый фронт будет закрыт. И удалить из всех гридов
    # это поле.
    grid.access_structure['width'] = '100%'
    grid.access_structure['sorting'] = []
    grid.change_structure(plain_structure=None, already_parsed=grid.access_structure)
    with transaction.atomic():
        grid.save()
        grid.authors.add(author_of_changes)

        # наблюдатели нового грида
        ancestor = grid.nearest_parent
        if ancestor:
            qs = PageWatch.objects.filter(page=ancestor, is_cluster=True).exclude(user=author_of_changes.username)
            for page_watch in qs:
                PageWatch(page=grid, is_cluster=True, user=page_watch.user).save()
        PageWatch(page=grid, is_cluster=True, user=author_of_changes.username).save()

        page_signals.access_changed.send(sender=create_grid, page_list=[grid])

        # нотификация
        grid_created(author_of_changes=author_of_changes, now=now, grid=grid)

    favorites_logic.page_changed(grid, was_created=True)

    return grid


def grid_created(author_of_changes, now, grid):
    """
    Создать PageEvent описывающий создание Page.

    @type now: datetime.datetime
    @rtype: PageEvent
    """
    from wiki.pages.models import Revision

    event = PageEvent(
        event_type=PageEvent.EVENT_TYPES.create,
        author=author_of_changes,
        timeout=now + datetime.timedelta(minutes=5),
        page=grid,
    )
    event.meta = {'revision_id': Revision.objects.create_from_page(grid).id}
    event.save()
    return event
