
from datetime import date
from itertools import product

from django.contrib.auth import get_user_model
from django.utils.encoding import smart_text
from django.utils.translation import ugettext_lazy

from wiki.grids.filter.base import FilterError
from wiki.grids.filter.semantictree import Combinator, Condition, Value
from wiki.org import org_user

CHECK_FUNCTION = {
    'EQUAL': lambda v, a: v == a,
    'CONTAINS': lambda v, a: a in v,
    'LESS': lambda v, a: v < a,
}


class EvaluatorError(FilterError):

    message = NotImplemented

    def __init__(self, *args, **kwargs):
        message = self.message.format(*args)
        super(EvaluatorError, self).__init__(message, **kwargs)


class UnknownNodeError(EvaluatorError):
    # Translators: Ошибка -- неизвестный узел в разобранном параметре filter: {0!r}
    message = ugettext_lazy('grids.Filter.UnknownNode {0!r}')


class UnknownConditionOpError(EvaluatorError):
    # Translators: Ошибка -- неизвестный оператор условия в разобранном параметре filter: {0!r}
    message = ugettext_lazy('grids.Filter.UnknownConditionOperator {0!r}')


class UnknownCombinatorOpError(EvaluatorError):
    # Translators: Ошибка -- неизвестный комбинатор в разобранном параметре filter: {0!r}
    message = ugettext_lazy('grids.Filter.UnknownCombinatorOperator {0!r}')


class NoColumnError(EvaluatorError):
    # Translators: Ошибка -- в табличном списке нет столбца с таким ID: {0!r}
    message = ugettext_lazy('grids.Filter.NoSuchColumn {0!r}')


class NotAValueError(EvaluatorError):
    # Translators: Ошибка -- плохой аргумент справа от {0} в разобранном параметре filter: {1!r}
    message = ugettext_lazy('grids.Filter.BadArg {0} {1!r}')


class ArgCastError(EvaluatorError):
    # Translators: Ошибка -- не удалось привести {0!r} к типу {1} в разобранном параметре filter
    message = ugettext_lazy('grids.Filter.CannotCastArg {0!r} {1}')


class UnknownArgType(EvaluatorError):
    # Translators: Ошибка -- неизвестный тип аргумента в разобранном параметре filter: {0}
    message = ugettext_lazy('grids.Filter.UnknownArgType {0}')


class _CastError(EvaluatorError):
    message = ''


def evaluate(node, grid, hashes=None):
    """
    @type grid: Grid
    """
    if node is None:
        return list(grid.access_idx), []
    assert_columns_exist(node, grid)
    hashes_in = []
    hashes_out = []
    for row_id, row in grid.row_id_and_row():
        if hashes and row_id not in hashes:
            continue
        if eval_row(node, row):
            hashes_in.append(row_id)
        else:
            hashes_out.append(row_id)
    return hashes_in, hashes_out


def eval_row(node, row):
    if isinstance(node, Condition):
        return eval_condition(node, row)
    elif isinstance(node, Combinator):
        return eval_combinator(node, row)
    else:
        raise UnknownNodeError(node)


def eval_condition(condition, row):
    if condition.operator not in CHECK_FUNCTION:
        raise UnknownConditionOpError(condition)
    if not all(isinstance(arg, Value) for arg in condition.args):
        raise NotAValueError(condition.operator, condition)
    values = get_values(row, condition.column)
    if condition.operator == 'EQUAL':
        return equal(values, condition.args, condition.negated)
    elif condition.operator == 'CONTAINS':
        return contains(values, condition.args, condition.negated)
    elif condition.operator == 'LESS':
        return less(values, condition.args, condition.negated)
    else:
        raise UnknownConditionOpError(condition)


def get_values(row, column_name):
    if column_name not in row:
        raise NoColumnError(column_name)
    value_info = row[column_name]
    if isinstance(value_info, dict):
        values = value_info['raw']
    else:
        values = value_info
    if not isinstance(values, list):
        values = [values]
    return values


def equal(values, args, is_negated):
    result = False
    if len(values) == 0 and len(args) == 0:
        result = True
    for (value, arg) in product(values, args):
        result = check_using_arg_type('EQUAL', value, arg)
        if result is None:
            result = check_using_strings('EQUAL', value, arg)
        if result is True:
            break
    if is_negated:
        result = not result
    return result


def contains(values, args, is_negated):
    result = False
    if len(values) == 0 and len(args) == 0:
        result = True
    for (value, arg) in product(values, args):
        result = check_using_strings('CONTAINS', value, arg)
        if result is True:
            break
    if is_negated:
        result = not result
    return result


def less(values, args, is_negated):
    result = False
    for (value, arg) in product(values, args):
        result = check_using_arg_type('LESS', value, arg)
        if result is True and not is_negated:
            break
        if result is False and is_negated:
            break
    if is_negated and result is not None:
        result = not result
    return result


def check_using_arg_type(operator, value, arg):
    try:
        arg_a = cast(arg.value, arg.type)
    except Exception:
        raise ArgCastError(arg.value, arg.type)
    try:
        value_a = cast(value, arg.type)
    except _CastError:
        return None
    return check(operator, value_a, arg_a)


def check_using_strings(operator, value, arg):
    arg_s = smart_text(arg.value).strip().lower()
    value_s = smart_text(value).strip().lower()
    return check(operator, value_s, arg_s)


def cast(value, arg_type):

    if isinstance(value, str):
        value = value.strip()

    if arg_type == 'NONE':
        return value if value else None

    elif arg_type == 'BOOLEAN':
        return bool(value)

    elif arg_type == 'DATE':
        try:
            return date(*[int(x) for x in value.split('-')])
        except (AttributeError, TypeError, ValueError):
            raise _CastError()

    elif arg_type == 'NUMBER':
        try:
            return float(value)
        except (TypeError, ValueError):
            raise _CastError()

    elif arg_type == 'STRING':
        if value is None:
            return ''
        else:
            try:
                return smart_text(value).strip().lower()
            except (TypeError, ValueError):
                raise _CastError()

    elif arg_type == 'USER':
        try:
            return org_user().get(username=value)
        except get_user_model().DoesNotExist:
            raise _CastError()

    elif arg_type == 'TICKET':
        return value

    else:
        raise UnknownArgType(arg_type)


def check(operator, v, a):
    try:
        return CHECK_FUNCTION[operator](v, a)
    except TypeError:
        return None


def eval_combinator(combinator, row):
    if combinator.operator == 'AND':
        result = all(eval_row(node, row) for node in combinator.args)
    elif combinator.operator == 'OR':
        result = any(eval_row(node, row) for node in combinator.args)
    else:
        raise UnknownCombinatorOpError(combinator)
    if combinator.negated:
        result = not result
    return result


def assert_columns_exist(node, grid, _column_names=None):
    if _column_names is None:
        _column_names = [field['name'] for field in grid.columns_meta()]
    if isinstance(node, Condition):
        if node.column not in _column_names:
            raise NoColumnError(node.column)
    elif isinstance(node, Combinator):
        for subnode in node.args:
            assert_columns_exist(subnode, grid, _column_names)
