
import ply.lex as lex  # See http://www.dabeaz.com/ply/ply.html
from django.utils.encoding import smart_str
from django.utils.translation import ugettext

from wiki.grids.filter.base import FilterError
from wiki.grids.filter.semantictree import Value


class LexerError(FilterError):
    pass


RESERVED_WORDS = '''
    EMPTY
    NULL
    NOT
    AND
    OR
    IN
    IS
    BETWEEN

    YES
    ON
    TRUE
    CHECKED
    DONE

    NO
    OFF
    FALSE
    UNCHECKED
    '''.split()

tokens = (
    '''
    COMMA
    WORD
    COLUMN
    LEFT_PAREN
    RIGHT_PAREN
    EQUAL
    NOT_EQUAL
    CONTAINS
    NOT_CONTAINS
    GREATER
    LESS
    GREATER_OR_EQUAL
    LESS_OR_EQUAL
    DATE
    NUMBER
    STRING
    USER
    TICKET
    '''.split()
    + RESERVED_WORDS
)

t_COMMA = r','
t_LEFT_PAREN = r'\('
t_RIGHT_PAREN = r'\)'
t_EQUAL = r'==?'
t_NOT_EQUAL = r'!='
t_CONTAINS = r'~'
t_NOT_CONTAINS = r'!~'
t_GREATER = r'>'
t_LESS = r'<'
t_GREATER_OR_EQUAL = r'>='
t_LESS_OR_EQUAL = r'<='


def t_COLUMN(t):
    r'\[[^\]]+\]'
    t.value = t.value[1:-1]
    return t


def t_DATE(t):
    r'\d{4}-\d{2}-\d{2}'
    t.value = Value('DATE', t.value)
    return t


def t_NUMBER(t):
    r'[+-]?\d+(\.\d+)?'
    t.value = Value('NUMBER', t.value)
    return t


def t_STRING(t):
    r"'[^']*'"
    t.value = Value('STRING', t.value[1:-1])
    return t


def t_USER(t):
    r'[\w-]+@'
    t.value = Value('USER', t.value[:-1])
    return t


def t_TICKET(t):
    r'\w+-\d+'
    t.value = Value('TICKET', t.value)
    return t


def t_WORD(t):
    r'[\wАБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдеёжзийклмнопрстуфхцчшщъыьэюя]+'
    upper = t.value.upper()
    if upper in RESERVED_WORDS:
        t.value = t.type = upper
    else:
        t.type = 'STRING'
        t.value = Value(type='STRING', value=t.value)
    return t


# Define a rule so we can track line numbers
def t_newline(t):
    r'\n+'
    t.lexer.lineno += len(t.value)


# A string containing ignored characters (spaces and tabs)
t_ignore = ' \t'


# Error handling rule
def t_error(t):
    # Translators: Ошибка -- плохой символ в параметре filter: {0}
    message_template = ugettext('grids.Filter.BadCharacter "{0}"')
    raise LexerError(smart_str(message_template).format(smart_str(t.value[0])))


# Build the lexer
lexer = lex.lex()
