
import pickle

import ply.yacc as yacc  # See http://www.dabeaz.com/ply/ply.html
from django.conf import settings
from django.core.cache import caches
from django.utils.encoding import smart_text
from django.utils.translation import ugettext

from wiki.grids.filter.base import FilterError
from wiki.grids.filter.lexer import tokens  # noqa -- испольуются в ply для построения самого парсера
from wiki.grids.filter.lexer import lexer
from wiki.grids.filter.semantictree import AND, CONTAINS, EQUAL, LESS, NOT, OR, Combinator, Value
from wiki.legacy.wf_compat import compat_pickle_loads


class ParserError(FilterError):
    pass


def p_xtree(p):
    """

    xtree : tree
          | nothing

    """
    p[0] = p[1]


def p_tree(p):
    """

    tree : tree COMMA xcondition
         | tree AND xcondition
         | tree OR xcondition
         | xcondition

    """
    if len(p) == 2:
        p[0] = p[1]
    else:
        if p[2] in ('AND', ','):
            if isinstance(p[1], Combinator) and p[1].operator == 'AND':
                p[1].args.append(p[3])
                p[0] = p[1]
            else:
                p[0] = AND(p[1], p[3])
        elif p[2] == 'OR':
            if isinstance(p[1], Combinator) and p[1].operator == 'OR':
                p[1].args.append(p[3])
                p[0] = p[1]
            else:
                p[0] = OR(p[1], p[3])


def p_xcondition(p):
    """

    xcondition : condition
               | NOT condition
               | LEFT_PAREN tree RIGHT_PAREN

    """
    if len(p) == 2:
        p[0] = p[1]
    elif p[1] == 'NOT':
        p[0] = NOT(p[2])
    else:
        p[0] = p[2]


def p_condition(p):
    """

    condition : column EQUAL value
              | column NOT_EQUAL value
              | column CONTAINS value
              | column NOT_CONTAINS value
              | column GREATER value
              | column LESS value
              | column GREATER_OR_EQUAL value
              | column LESS_OR_EQUAL value
              | column IS value
              | column IS NOT value
              | column IN xvalueslist
              | column NOT IN xvalueslist
              | column BETWEEN value AND value

    """
    if p[2] in ('=', '=='):
        p[0] = EQUAL(p[1], [p[3]])
    elif p[2] == '!=':
        p[0] = NOT(EQUAL(p[1], [p[3]]))
    elif p[2] == '~':
        p[0] = CONTAINS(p[1], [p[3]])
    elif p[2] == '!~':
        p[0] = NOT(CONTAINS(p[1], [p[3]]))
    elif p[2] == '<':
        p[0] = LESS(p[1], [p[3]])
    elif p[2] == '<=':
        p[0] = OR(LESS(p[1], [p[3]]), EQUAL(p[1], [p[3]]))
    elif p[2] == '>=':
        p[0] = NOT(LESS(p[1], [p[3]]))
    elif p[2] == '>':
        p[0] = AND(NOT(LESS(p[1], [p[3]])), NOT(EQUAL(p[1], [p[3]])))
    elif p[2] == 'IS' and p[3] == 'NOT':
        p[0] = NOT(EQUAL(p[1], [p[4]]))
    elif p[2] == 'IS':
        p[0] = EQUAL(p[1], [p[3]])
    elif p[2] == 'IN':
        p[0] = EQUAL(p[1], p[3])
    elif p[2] == 'NOT' and p[3] == 'IN':
        p[0] = NOT(EQUAL(p[1], p[4]))
    elif p[2] == 'BETWEEN' and p[4] == 'AND':
        p[0] = OR(EQUAL(p[1], [p[3]]), EQUAL(p[1], [p[5]]), AND(NOT(LESS(p[1], [p[3]])), LESS(p[1], [p[5]])))


def p_column(p):
    """

    column : COLUMN

    """
    p[0] = p[1]


def p_value(p):
    """

    value : DATE
          | NUMBER
          | STRING
          | USER
          | TICKET
          | YES
          | ON
          | TRUE
          | CHECKED
          | DONE
          | NO
          | OFF
          | FALSE
          | UNCHECKED
          | NULL
          | EMPTY

    """
    if p[1] == 'NULL' or p[1] == 'EMPTY':
        p[0] = Value(type='NONE', value='')
    elif p[1] in ('YES', 'ON', 'TRUE', 'CHECKED', 'DONE'):
        p[0] = Value(type='BOOLEAN', value=True)
    elif p[1] in ('NO', 'OFF', 'FALSE', 'UNCHECKED'):
        p[0] = Value(type='BOOLEAN', value=False)
    else:
        p[0] = p[1]


def p_xvalueslist(p):
    """

    xvalueslist : LEFT_PAREN valueslist RIGHT_PAREN
                | LEFT_PAREN nothing RIGHT_PAREN

    """
    if p[2] is None:
        p[0] = []
    else:
        p[0] = p[2][:]


def p_valueslist(p):
    """

    valueslist : value COMMA valueslist
               | value

    """
    if len(p) > 2:
        p[0] = [p[1]] + p[3][:]
    else:
        p[0] = [p[1]]


def p_nothing(p):
    """

    nothing :

    """


# Error rule for syntax errors
def p_error(p):
    if hasattr(p, 'value'):
        if isinstance(p.value, Value):
            value_str = '{0} {1!r}'.format(p.value.type, p.value.value)
        else:
            value_str = repr(p.value)
    else:
        value_str = repr(p)
    if p is None:
        message = (
            # Translators: Ошибка -- неожиданный конец текста в параметре filter
            ugettext('grids.Filter.ParsingErrorAtEnd')
        )
    elif hasattr(p, 'lexpos'):
        message = (
            # Translators: Ошибка рядом с {0} начиная с символа номер {1} в параметре filter
            ugettext('grids.Filter.ParsingErrorAt {0} {1}').format(value_str, p.lexpos + 1)
        )
    else:
        message = (
            # Translators: Ошибка рядом с {0} в параметре filter
            ugettext('grids.Filter.ParsingErrorNear {0}').format(value_str)
        )
    raise ParserError(message)


parser_options = {}

# Указать при создании парсера где в файловой системе сохранять таблицу парсинга
if hasattr(settings, 'PARSER_TABLE_PATH'):
    parser_options['outputdir'] = settings.PARSER_TABLE_PATH
    parser_options['tabmodule'] = 'grids_filter_tab'
else:
    parser_options['write_tables'] = 0

# Отключить ворнинги
if getattr(settings, 'SUPPRESS_PARSER_WARNINGS', True):
    parser_options['errorlog'] = yacc.NullLogger()

# Build the parser
parser = yacc.yacc(debug=1, **parser_options)


def parse(text):
    text = smart_text(text)
    tree = get_tree_from_cache(text)
    if tree is None:
        tree = parser.parse(text, lexer=lexer)
        put_tree_in_cache(text, tree)
    return tree


def get_tree_from_cache(text):
    cache = caches['grid_filter']
    pickled_tree = cache.get(make_cache_key(text))
    if pickled_tree is None:
        return None
    try:
        return compat_pickle_loads(pickled_tree)
    except pickle.PickleError:
        return None


def put_tree_in_cache(text, tree):
    cache = caches['grid_filter']
    try:
        pickled_tree = pickle.dumps(tree)
    except pickle.PickleError:
        pass
    else:
        cache.set(make_cache_key(text), pickled_tree)


def make_cache_key(text):
    return str(hash(text))
