
import uuid
from copy import deepcopy
from hashlib import sha256
from random import randint

from django import forms
from django.core.cache import caches
from django.db.models import Q
from django.utils.translation import ngettext, ugettext, ugettext_lazy

from wiki.grids.utils import make_beautiful_structure_from, the_field_types
from wiki.org import org_staff
from wiki.utils.supertag import translit

try:
    from ujson import dumps, loads  # noqa
except ImportError:
    from json import dumps, loads  # noqa

cache_source = caches['imported_grids']


class FileUploadForm(forms.Form):
    """Джанго-форма для загрузки файла в табличный список."""

    ALLOWED_EXTENSIONS = {'csv': ''}
    grid_data = forms.FileField(
        required=True,
        error_messages={
            'unknown_filetype': ugettext_lazy('grids.clone:Filename is wrong'),
            'wrong_filetype': ugettext_lazy('grids.clone:The type of file is wrong'),
        },
    )

    def clean_grid_data(self):
        value = self.cleaned_data.get('grid_data')
        filename_parts = value.name.split('.')
        field = self.fields['grid_data']
        if len(filename_parts) < 2:
            raise forms.ValidationError(field.error_messages['unknown_filetype'])
        if filename_parts[-1] not in list(self.ALLOWED_EXTENSIONS.keys()):
            raise forms.ValidationError(field.error_messages['wrong_filetype'])
        return value

    def save(self):
        the_file = self.cleaned_data.get('grid_data')
        key = sha256(translit(the_file.name) + str(uuid.uuid4())).hexdigest()
        cache_source.set(
            key,
            {
                'name': the_file.name,
                'contents': the_file.read(),
            },
        )
        return key


class TuneCsvForm(forms.Form):
    """Джанго-форма для настройки чтения из загруженного файла"""

    ENCODING_CHOICE = (
        ('cp1251', ugettext_lazy('grids.import:Windows-1251')),
        ('utf-8', ugettext_lazy('grids.import:UTF-8')),
    )
    DELIMITER_CHOICE = (
        (';', ugettext_lazy('grids.import:Excel default')),
        (',', ugettext_lazy('grids.import:Comma')),
        ('\t', ugettext_lazy('grids.import:Tab')),
    )
    QUOTECHAR_CHOICE = (
        ('"', ugettext_lazy('grids.import:Doublequote Excel default')),
        ("'", ugettext_lazy('grids.import:Singlequote')),
    )
    charset = forms.CharField(required=True, initial='cp1251', widget=forms.Select(choices=ENCODING_CHOICE))
    delimiter = forms.CharField(required=True, initial=';', widget=forms.Select(choices=DELIMITER_CHOICE))
    quotechar = forms.CharField(required=True, initial='"', widget=forms.Select(choices=QUOTECHAR_CHOICE))
    omit_first = forms.BooleanField(required=True, initial=False)


CHOICES = list(the_field_types.items())

# TODO: вернуть эти варианты
# CHOICES.append(("multiple_staff", ugettext_lazy("grids.import:Multiple staff")))
# CHOICES.append(("multiple_choice", ugettext_lazy("grids.import:Multiple choice")))

is_column = 'column_'


def grid_structure(grid):
    """
    Упростить структуру грида, чтобы добавить её в поля формы для выбора

    @type grid: Grid
    """
    choices = []
    cnt = 1
    for column in grid.columns_by_type(*list(the_field_types.keys())):
        title = column['title']
        if not title:
            _type = the_field_types.get(column['type']) or ugettext('grids:of unknown type')
            title = ngettext('grid.import:first column %s', 'grid.import:column number %s', cnt) % cnt
            title += ' (%s)' % str(_type)
        choices.append({'name': is_column + column['name'], 'title': title, 'type': column['type']})
        cnt += 1
    return choices


def is_between_field_types(column_type):
    return column_type in dict(CHOICES)


CHECKBOX_PREFIX = 'import'
COLUMN_PREFIX = 'to'


class ParseFileForm(forms.Form):
    """Джанго-форма для сохранения загруженного файла в грид."""

    def __init__(self, reader, grid, data=None, *args, **kwargs):
        choices = []
        if grid.pk:
            for choice in grid_structure(grid):
                choices.append((choice['name'], choice['title']))
        self.grid_choices = list(choices)
        choices += CHOICES

        self.reader = reader
        self.reader.open()

        for column in range(self.reader.column_count):
            checkbox_name = CHECKBOX_PREFIX + str(column)
            column_name = COLUMN_PREFIX + str(column)
            if data:
                column_initial = data.get(column_name)
                checkbox_initial = data.get(checkbox_name)
            else:
                column_initial = None
                checkbox_initial = None
            self.base_fields[checkbox_name] = forms.BooleanField(required=False, initial=checkbox_initial)
            self.base_fields[column_name] = forms.ChoiceField(choices=choices, required=False, initial=column_initial)

        super(ParseFileForm, self).__init__(data, *args, **kwargs)
        self.grid = grid

    def clean(self):
        unique_columns = set([])
        i = -1
        while True:
            i += 1
            column_name = COLUMN_PREFIX + str(i)
            if column_name not in self.fields:
                break
            checkbox_name = CHECKBOX_PREFIX + str(i)
            if not self.cleaned_data.get(checkbox_name):
                continue
            value = self.cleaned_data.get(column_name)
            if not value or not value.startswith(is_column):
                continue
            if value in unique_columns:
                raise forms.ValidationError(ugettext('grids.import:Duplicate column'))
            unique_columns.add(value)
        return self.cleaned_data

    def save(self, import_data=False, **kwargs):
        grid = self.grid
        if grid.pk:
            old_structure = {'structure': deepcopy(grid.access_structure)}
        else:
            old_structure = {'structure': {'fields': []}}
        grid_fields = old_structure['structure']['fields']

        column_grid_map = {}
        column_count = 1
        i = -1
        while True:
            i += 1
            checkbox_name = CHECKBOX_PREFIX + str(i)
            if checkbox_name not in self.fields:
                break
            if not self.cleaned_data.get(checkbox_name):
                continue
            column_name = COLUMN_PREFIX + str(i)
            column_type = self.cleaned_data.get(column_name)
            if is_between_field_types(column_type):
                grid_column_name = str(randint(100, 1000))
            else:
                grid_column_name = column_type[len(is_column) :]

            column_grid_map[i] = grid_column_name

            if not (isinstance(column_type, str) and is_between_field_types(column_type)):
                continue

            column_description = {
                'type': column_type,
                'name': grid_column_name,
                'title': ugettext('grids.import:column %s') % column_count,
            }

            if column_type in ('multiple_staff', 'multiple_select'):
                column_description['multiple'] = True
                column_description['type'] = column_description['type'][len('multiple_') :]
            if column_type in ('select', 'multiple_select'):
                column_description['options'] = []
            grid_fields.append(column_description)
            column_count += 1

        if column_count > 1:
            new_structure = make_beautiful_structure_from(dumps(old_structure))['structure']
            grid.change_structure(None, new_structure)

        inserted_rows = []
        if import_data:
            from wiki.grids.logic.grids_import import import_data_to_grid

            inserted_rows = import_data_to_grid(grid, self.reader, column_grid_map, import_data, **kwargs)

        # заполнить поля типа select опциями

        return inserted_rows, column_count > 1


def options_extractor(select_map):
    """
    Наполняет select_map уникальными значениями
    select_map = defaultdict(set)
    """

    def wrap(iterable):
        for row in iterable:
            for name in row:
                if name not in select_map or not row[name]:
                    continue
                value = row[name]
                if not isinstance(value, str) and hasattr(value, '__iter__'):
                    list(map(lambda v: select_map[name].add(v.strip()), value))
                else:
                    value = value.strip()
                    if not value:
                        continue
                    select_map[name].add(value.strip())
            yield row

    return wrap


def recognize_staff(staff_field_names):
    def do_recognize(string):
        str1, str2 = string.split(' ', 2)
        condition = Q(first_name__iexact=str1, last_name__iexact=str2) | Q(
            first_name__iexact=str2, last_name__iexact=str1
        )
        result = org_staff().filter(condition)
        if len(result) == 1:
            return result[0]
        return False

    def get_login_by_name(string):
        if ' ' not in string:
            return string
        try:
            staff = do_recognize(string)
        except Exception:
            return string
        if not staff:
            return string
        return staff.login

    def wrap(iterable):
        for row in iterable:
            for name in row:
                if name not in staff_field_names or not row[name]:
                    continue
                values = [val.strip() for val in row[name].strip().split(',')]

                row[name] = ','.join(map(get_login_by_name, values))
            yield row

    return wrap
