

import logging

from wiki.grids.logic.tickets import ticket_dependent_types

logger = logging.getLogger(__name__)


def column_by_name(grid, name):
    """
    Вернуть мета-информацию столбца по его имени или None.

    @rtype: dict
    """
    return column_by_name_from_access_structure_fields(grid.columns_meta(), name)


def column_by_name_from_access_structure_fields(fields, name):
    """
    Вернуть мета-информацию столбца по его имени или None.

    @rtype: dict
    """
    for index, column in enumerate(fields):
        if name == column['name']:
            return column


def columns_by_type(grid, *column_types):
    """
    Вернуть столбцы заданных типов.

    Если column_types не задано, то результат будет "все столбцы".

    @type grid: Grid
    @rtype: dict
    """
    if not column_types:
        logger.debug('No column types in filter for "%s"', grid)
        for column in grid.columns_meta():
            yield column
        return
    column_types = set(column_types)
    for index, _type, column in grid.columns():
        if _type in column_types:
            yield column


def column_names_by_type(grid, *column_types):
    """
    Вернуть имена заданных столбцов по типу.

    Бывает полезна, если вам надо поработать со строками:
      * переформатировать часть
      * посмотреть содержимое части
    """
    for column in columns_by_type(grid, *column_types):
        yield column['name']


def column_attribute_by_names(grid, attr, *names):
    """
    Вывести нужные атрибуты столбцов, столбцы заданы именами.

    Если есть имена столбцов и надо вывести, например, человекочитаемые названия.

    @type attr: basestring

    Если столбца нет, то бросает KeyError, если атрибута нет, то бросает KeyError.
    """
    for name in names:
        column = column_by_name(grid, name)
        if not column:
            raise KeyError
        yield column[attr]


def remove_column(grid, column_name):
    """
    Удалить столбец по имени.

    @type grid: Grid
    @raises: ValueError
    """
    column = grid.column_by_name(column_name)
    if not column:
        return grid
    if column['type'] == 'ticket':  # нельзя удалять если есть ticket_dependent столбцы
        if list(grid.column_names_by_type(*ticket_dependent_types)):
            # надо сперва удалить стобцы которые зависят от грида
            raise ValueError('Grid has ticket dependent columns')
    new_structure = grid.access_structure.copy()
    new_structure['fields'].remove(column)
    new_data = list(grid.access_data)
    for row in new_data:
        if column_name in row:
            del row[column_name]
    grid.access_structure = new_structure
    grid.access_data = new_data
    return grid
