
from collections import OrderedDict

from wiki.grids.logic.structure import GridWrapper


def build_diff(old_grid, new_grid):
    old_grid, new_grid = GridWrapper(old_grid), GridWrapper(new_grid)
    diff_matrix = build_diff_structure(old_grid, new_grid)

    for row_key, row in diff_matrix.items():
        for column_key, cell in row.items():
            new_value = new_grid.get((row_key, column_key))
            old_value = old_grid.get((row_key, column_key))
            new_value = new_value and new_value['raw']
            old_value = old_value and old_value['raw']

            was_in_old_grid = (row_key, column_key) in old_grid
            is_in_new_grid = (row_key, column_key) in new_grid
            if not was_in_old_grid and not is_in_new_grid:
                continue
            elif not was_in_old_grid:
                diff_matrix[row_key][column_key] = {'new': new_value}
            elif not is_in_new_grid:
                diff_matrix[row_key][column_key] = {'old': old_value}
            elif new_value != old_value:
                diff_matrix[row_key][column_key] = {
                    'old': old_value,
                    'new': new_value,
                }
            else:
                diff_matrix[row_key][column_key] = {'current': new_value}

    # remove unchanged rows
    row_keys = list(diff_matrix.keys())

    for row_key in row_keys:
        row = diff_matrix[row_key]
        if all(val and 'current' in val for val in list(row.values())):
            del diff_matrix[row_key]

    titles_diff = build_diff_column_titles(old_grid, new_grid, diff_matrix)

    return diff_matrix, titles_diff


def build_diff_structure(old_grid, new_grid):
    """
    Возвращает заготовку матрицы, которая представляет собой дифф гридов.
    """
    rows_order, columns_order = _calculate_diff_matrix_structure(
        old_rows_order=old_grid.rows_keys,
        old_columns_order=old_grid.columns_keys,
        new_rows_order=new_grid.rows_keys,
        new_columns_order=new_grid.columns_keys,
    )

    matrix = OrderedDict()
    for row_key in rows_order:
        row_cells = OrderedDict()
        matrix[row_key] = row_cells
        for column_key in columns_order:
            row_cells[column_key] = None

    return matrix


def build_diff_column_titles(old_grid, new_grid, diff_matrix):
    titles_diff = OrderedDict()

    first_row = diff_matrix and list(diff_matrix.values())[0] or []
    for column_key in first_row:
        was_in_old_grid = column_key in old_grid.columns_keys
        is_in_new_grid = column_key in new_grid.columns_keys
        old_value = old_grid.get_field_by_key(column_key)
        new_value = new_grid.get_field_by_key(column_key)
        old_value = old_value and old_value['title']
        new_value = new_value and new_value['title']
        if not was_in_old_grid:
            titles_diff[column_key] = {'new': new_value}
        elif not is_in_new_grid:
            titles_diff[column_key] = {'old': old_value}
        elif new_value != old_value:
            titles_diff[column_key] = {
                'old': old_value,
                'new': new_value,
            }
        else:
            titles_diff[column_key] = {'current': new_value}

    return titles_diff


def _calculate_diff_matrix_structure(
    old_rows_order,
    old_columns_order,
    new_rows_order,
    new_columns_order,
):
    return (
        _smart_merge_key_lists(old_rows_order, new_rows_order),
        _smart_merge_key_lists(old_columns_order, new_columns_order),
    )


def _smart_merge_key_lists(iterable_one, iterable_two):
    """
    Смержить два списка уникальных ключей в один список, содержащий все ключи.
    Ключ из первого списка, которого нет во втором пытаемся засунуть
    куда-то перед элементом, перед которым он стоит в первом списке.

    Используем, чтобы более-менее логично расставить столбцы или строки в
    диффе гридов.
    """
    merged_iterable = list(iterable_two)

    for index, row_key in enumerate(iterable_one):
        if row_key in merged_iterable:
            continue

        tail = iterable_one[index + 1 :]
        nearest_alive_key = _get_first_common_element(tail, merged_iterable)
        if nearest_alive_key is None:
            merged_iterable.append(row_key)
        else:
            nearest_alive_key_idx = merged_iterable.index(nearest_alive_key)
            merged_iterable.insert(nearest_alive_key_idx, row_key)

    return merged_iterable


def _get_first_common_element(iterable_one, iterable_two):
    """
    Ищем первый элемент в iterable_one, который есть в iterable_two.

    Используем, чтобы найти ближайший ключ строки или столбца из старой
    структуры, чтобы найти подходящее место в диффе.
    """
    for item in iterable_one:
        if item not in iterable_two:
            continue
        return item
