from typing import Dict, List

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.grids.logic.grids_import import MAX_LINES_TO_IMPORT, InvalidImportDataError, import_data_to_grid
from wiki.grids.models import Grid, Revision, timezone
from wiki.grids.readers.base import BaseReader
from wiki.grids.utils import RowValidationError


class BadColumns(InvalidDataSentError):
    error_code = 'BAD_COLUMNS_REFERENCES'
    debug_message = 'Non-existing grid column referenced.'


class JsonReader(BaseReader):
    def __init__(self, obj: List[Dict[str, str]], column_order: List[str]):
        self.obj = obj
        self.column_order = column_order

    def open(self):
        return self

    def __iter__(self):
        for row in self.obj:
            yield [row.get(column_name, '') for column_name in self.column_order]

    @property
    def row_count(self) -> int:
        return len(self.obj)

    @property
    def column_count(self):
        if self.row_count == 0:
            return 0
        return len(self.obj[0].keys())


def replace_grid_data(request, grid: Grid, rows: List[Dict[str, str]]):
    """
    rows: { 'column_id': 'value' }
    :param grid:
    :param rows:
    :return:
    """
    referenced_columns = set()
    for row in rows:
        referenced_columns.update((row.keys()))

    columns = grid.access_structure['fields']
    available_columns = set([c['name'] for c in columns])

    if len(referenced_columns - available_columns) > 0:
        raise BadColumns()

    grid.access_idx.clear()
    grid.access_data.clear()
    grid.access_meta['autoincrement'] = 0

    params = {'request': request}

    try:
        import_data_to_grid(
            grid,
            JsonReader(rows, list(referenced_columns)),
            {idx: col for idx, col in enumerate(referenced_columns)},
            MAX_LINES_TO_IMPORT,
            **params
        )
    except RowValidationError as e:
        raise InvalidImportDataError(errors=e.errors)

    grid.modified_at = timezone.now()

    if request.user.id:
        grid.last_author = request.user
    grid.save()

    Revision.objects.create_from_page(grid)
    return grid
