import logging
from collections import defaultdict
from copy import deepcopy
from datetime import timedelta
from random import randint
from typing import Dict

import chardet
from django.conf import settings
from django.core.cache import caches
from django.utils.translation import ugettext

from wiki.grids.forms import grid_structure, is_column, options_extractor, recognize_staff
from wiki.grids.models import Grid, Revision
from wiki.grids.readers import mapped_reader, reader
from wiki.grids.readers.base import BaseReader
from wiki.grids.utils import (
    HASH_KEY,
    RowValidationError,
    changes_of_structure,
    grid_as_table,
    insert_rows,
    make_beautiful_structure_from,
)
from wiki.grids.utils.base import STAFF_TYPE, insert_missing_fields, the_field_types
from wiki.notifications.models import PageEvent
from wiki.org import get_org
from wiki.utils import timezone

try:
    from ujson import dumps, loads  # noqa
except ImportError:
    from json import dumps, loads  # noqa

logger = logging.getLogger(__name__)

cache_source = caches['imported_grids']

MAX_LINES_TO_IMPORT = 10000
MAX_ROWS_TO_FORMAT = 251
MAX_LINES_TO_PARSE_PREVIEW = 10

DEFAULT_TIMEOUT = timezone.now() + timedelta(minutes=20)

IMPORT_COLUMN_NAME_PREFIX = 'icolumn_'
IMPORT_CHECKBOX_NAME_POSTFIX = '_enabled'
IMPORT_TO_NAME_POSTFIX = '_to'
IMPORT_TYPE_NAME_POSTFIX = '_type'

DEFAULT_FILE_DATA_FORMAT_PARAMS = {
    'delimiter': (';', ',', '\t'),
    'quotechar': ('\"', '\''),
}

VALID_IMPORT_FILE_EXTENTIONS = ['xls', 'xlsx', 'csv', 'txt']


class InvalidImportDataError(Exception):
    def __init__(self, errors=None, detail=None):
        self.errors = errors
        self.detail = detail


class ParseDataPreview(object):
    """
    Используется для хранения результата разбора файла с импортируемыми данными.
    """

    def __init__(self, cache_key, column_count=0, row_count=0, rows=None):
        self.cache_key = cache_key
        self.column_count = column_count
        self.row_count = row_count
        self.rows = rows or list()


class ImportDataPreview(object):
    """
    Используется для хранения результатов импорта данных в табличный список.
    """

    def __init__(
        self,
        grid,
        cache_key,
        column_count=0,
        row_count=0,
    ):
        self.cache_key = cache_key
        self.column_count = column_count
        self.row_count = row_count
        self.import_columns = []

        self.existing_rows = []
        self.imported_rows = []
        self.fields = []

        # список типов данных для нового столбца: [('string', 'Текст'), (...)]
        if settings.IS_INTRANET:
            self.column_data_types = list(the_field_types.items())
        else:
            _the_field_types = the_field_types.copy()
            _the_field_types.pop(STAFF_TYPE)
            self.column_data_types = list(_the_field_types.items())

        # словарь с именами импортируемых столбцов: ключ - индекс столбца в import_columns, value - имя столбца в новой
        # структуре грида (словарь используется при построении превью)
        self.grid_names_map = {}

        # список столбцов существующего гриду, куда можно импортировать:
        # [(column_name, column_title, column_type), (...)]
        self.existing_columns = []
        self.columns_map = {}
        if grid.pk:
            for choice in grid_structure(grid):
                column_name = choice['name'][len(is_column) :]
                self.existing_columns.append((column_name, choice['title'], choice['type']))
                self.columns_map[column_name] = (choice['title'], choice['type'])


def _get_http_params_value(request, param_name):
    if request.method == 'GET':
        return request.query_params.get(param_name)
    elif request.method == 'POST':
        return request.data.get(param_name, request.query_params.get(param_name))


def _get_http_bool_param_value(request, param_name, default_value=None):
    value = _get_http_params_value(request, param_name)

    if value is None:
        return default_value

    if isinstance(value, bool):
        return value

    if isinstance(value, str):
        if value.isdigit():
            return bool(int(value))
        elif value.lower() in ('true', 'false'):
            return value.lower() == 'true'

    return default_value


def _get_format_params(request):
    format_params = {}
    get_param_value_func = lambda val, values: val if val in values else values[0]

    for key, available_values in DEFAULT_FILE_DATA_FORMAT_PARAMS.items():
        format_params[key] = get_param_value_func(_get_http_params_value(request, key), available_values)

    format_params['omit_first'] = int(_get_http_bool_param_value(request, 'omit_first') or False)

    return format_params


def _get_contents_from_cache(request, cache_key):
    cache = cache_source.get(cache_key)

    if not cache:
        raise InvalidImportDataError(
            errors={'key': 'No content in cache for the specified key or the key is already expired'}
        )

    name, contents = cache['name'], cache['contents']

    encoding = _get_http_params_value(request, 'charset')
    if not encoding:
        try:
            encoding = chardet.detect(contents)['encoding']
        except (ValueError, KeyError) as err:
            logger.warn("Can't determine import file encoding: %s" % repr(err))
            raise InvalidImportDataError(
                detail="Can't determine import file encoding. Please specify the encoding explicitly"
            )

    if not encoding:
        raise InvalidImportDataError(detail='Invalid format of the file contents')

    try:
        contents = contents.decode(encoding)
    except UnicodeError as err:
        logger.warn("Can't encode import file content: encoding='%s' error:%s" % (encoding, repr(err)))
        raise InvalidImportDataError(detail="Can't encode import file content. Please try to specify another encoding")

    return name, contents


def _get_dict_key_by_value(items, lookup_value):
    for key, value in items:
        if value == lookup_value:
            return key


def _get_field_from_list_by_name(fields, name):
    for field in fields:
        if field['name'] == name:
            return field


def get_file_extention(file_name):
    try:
        return file_name.rsplit('.', 1)[1]
    except IndexError:
        return ''


def check_file_extention(extension):
    if extension not in VALID_IMPORT_FILE_EXTENTIONS:
        raise InvalidImportDataError(
            detail=ugettext('%s is unsupported file extension. Valid values: %s')
            % (extension, VALID_IMPORT_FILE_EXTENTIONS)
        )


def created_grid(user, grid):
    page_event = PageEvent(
        author=user, page=grid, event_type=PageEvent.EVENT_TYPES.create, notify=True, timeout=DEFAULT_TIMEOUT
    )
    page_event.meta = {
        'revision_id': Revision.objects.create_from_page(grid).id,
    }
    page_event.save()


def save_imported_grid(request, grid, inserted_rows, previous_structure, changed_structure, existing_page):
    grid.last_author = request.user
    grid.save()

    # создать событие и ревизию
    if not existing_page:
        grid.authors.add(grid.last_author)
        created_grid(request.user, grid)
    elif changed_structure:
        # сохранить структуру, чтобы потом построить diff по ней
        changes_of_structure(request.user, grid, previous_structure)

    # принудительно отключаю, иначе это пытка для БД
    # else:
    #     for row_id in inserted_rows:
    #         row_change_action[PageEvent.EVENT_TYPES.create](
    #             grid.get_rows(request.user_auth, [row_id])[0], request.user, grid
    #         )

    revision = Revision.objects.filter(page=grid, mds_storage_id=grid.mds_storage_id).first()
    if not revision:
        revision = Revision.objects.create_from_page(grid)


def import_data_to_grid(grid: Grid, grid_reader: BaseReader, column_map: Dict[int, str], max_rows, **kwargs):
    grid_reader.open()
    row_count = grid_reader.row_count
    if kwargs.get('no_lazy_format'):
        lazy_format = False
    else:
        lazy_format = row_count > MAX_ROWS_TO_FORMAT
    wrapper = insert_missing_fields(grid, max_rows)

    # поля типа select
    select_fields = defaultdict(set)
    staff_fields = []

    for field in grid.columns_by_type('select', 'staff'):
        if field.get('type') in ('select',):
            select_fields[field['name']]
        if field.get('type') == 'staff':
            staff_fields.append(field['name'])

    extractor_of_options = options_extractor(select_fields)

    staff_extractor = recognize_staff(staff_fields)

    provider_of_rows = wrapper(
        extractor_of_options(staff_extractor(mapped_reader(column_map, grid_reader, format_strings=not lazy_format)))
    )
    inserted_rows = insert_rows(grid, provider_of_rows, request=kwargs['request'], after='last')

    # нужно изменить структуру полей типа select
    struct = grid.access_structure
    fields = struct['fields']
    for field in fields:
        if field['name'] in select_fields:
            field['options'] = list(select_fields[field['name']])
    grid.access_structure = struct

    if lazy_format:
        meta = grid.access_meta
        meta['imported'] = True
        grid.access_meta = meta

    return inserted_rows


def _prepare_grid(request):
    if request.page:
        grid = Grid.objects.get(pk=request.page.pk)
    else:
        grid = Grid(
            owner=request.user,
            last_author=request.user,
            status=True,
            page_type=Grid.TYPES.GRID,
            tag=request.tag,
            supertag=request.supertag,
            org=get_org(),
        )

    return grid


def _get_grid(request):
    if request.page:
        grid = Grid.objects.get(pk=request.page.pk)
        return grid
    return None


def _get_file_reader(file_name):
    extension = file_name.split('.')[-1]
    check_file_extention(extension)
    file_reader = reader(extension)
    if not file_reader:
        raise InvalidImportDataError(detail=ugettext('Sorry. This file extension is not suppоrted yet.'))
    return file_reader


def do_import(request, cache_key, grid, for_preview=False):
    name, contents = _get_contents_from_cache(request, cache_key)

    grid_reader = _get_file_reader(name)(contents, **_get_format_params(request))
    grid_reader.open()

    if grid.pk:
        old_structure = {'structure': deepcopy(grid.access_structure)}
    else:
        old_structure = {'structure': {'fields': []}}

    grid_fields = []
    if not for_preview:
        grid_fields = old_structure['structure']['fields']
    column_grid_map = {}
    preview_data = ImportDataPreview(
        grid, cache_key, column_count=grid_reader.column_count, row_count=grid_reader.row_count
    )

    imported_column_count = 0
    for column_indx in range(preview_data.column_count):
        column_name = IMPORT_COLUMN_NAME_PREFIX + str(column_indx)
        import_to = _get_http_params_value(request, column_name + IMPORT_TO_NAME_POSTFIX)
        enabled = _get_http_bool_param_value(request, column_name + IMPORT_CHECKBOX_NAME_POSTFIX, True)
        if import_to:
            if import_to in preview_data.columns_map:
                column_type = preview_data.columns_map.get(import_to)[1]
                column_title = preview_data.columns_map.get(import_to)[0]
            else:
                raise InvalidImportDataError(errors={'import_to': 'Unknown column name "%s"' % import_to})
        else:
            column_type = _get_http_params_value(request, column_name + IMPORT_TYPE_NAME_POSTFIX) or 'string'
            column_title = ugettext('grids.import:column %s') % str(column_indx + 1)
        column = {
            'column_name': column_name,
            'title': column_title,
            column_name + IMPORT_CHECKBOX_NAME_POSTFIX: enabled,
            column_name + IMPORT_TO_NAME_POSTFIX: import_to,
            column_name + IMPORT_TYPE_NAME_POSTFIX: column_type,
        }
        preview_data.import_columns.append(column)

        if enabled:
            grid_column_name = import_to or str(randint(100, 1000))
            column_grid_map[column_indx] = grid_column_name

            if not import_to:
                # если это новый столбец, то добавим его в структуру грида
                imported_column_count += 1
                column_description = {
                    'type': column_type,
                    'name': grid_column_name,
                    'title': ugettext('grids.import:column %s') % imported_column_count,
                }

                if column_type in ('multiple_staff', 'multiple_select'):
                    column_description['multiple'] = True
                    column_description['type'] = column_description['type'][len('multiple_') :]
                if column_type in ('select', 'multiple_select'):
                    column_description['options'] = []
                grid_fields.append(column_description)
            elif for_preview:
                existing_column = grid.column_by_name(import_to)
                if existing_column:
                    grid_fields.append(existing_column)

    if for_preview:
        old_structure['structure']['fields'] = grid_fields
        preview_data.grid_names_map = column_grid_map

    if imported_column_count > 0:
        new_structure = make_beautiful_structure_from(dumps(old_structure))['structure']
        grid.change_structure(None, new_structure)

    try:
        params = {'request': request}
        if for_preview:
            params['no_lazy_format'] = True
            max_lines_to_import = MAX_LINES_TO_PARSE_PREVIEW
        else:
            max_lines_to_import = MAX_LINES_TO_IMPORT
        inserted_rows = import_data_to_grid(grid, grid_reader, column_grid_map, max_lines_to_import, **params)
    except RowValidationError as e:
        raise InvalidImportDataError(errors=e.errors)

    return inserted_rows, preview_data, imported_column_count > 0


def get_import_preview_data(request, cache_key, row_count_for_preview):
    grid = _prepare_grid(request)
    existing_hashes_to_show = []
    if grid.pk:
        # выбираем для превью хэши двух последних строк существующего грида, если они есть
        existing_hashes_to_show = [row[HASH_KEY] for row in grid.access_data[-row_count_for_preview:]]

    inserted_rows, preview_data, _ = do_import(request, cache_key, grid, True)

    # оставить в структуре грида для превью только те столбцы, которые enabled и данные из них нужны для превью
    grid_structure_for_preview = grid.access_structure.copy()
    grid_structure_for_preview['fields'] = []

    for imported_column_name in preview_data.grid_names_map.values():
        grid_structure_for_preview['fields'].append(
            _get_field_from_list_by_name(grid.access_structure['fields'], imported_column_name)
        )

    preview_data.fields, preview_data.imported_rows = grid_as_table(
        grid_structure_for_preview,
        grid.get_rows(request.user_auth, hashes=inserted_rows[:3]),
    )

    if existing_hashes_to_show:
        _, preview_data.existing_rows = grid_as_table(
            grid_structure_for_preview,
            grid.get_rows(request.user_auth, hashes=existing_hashes_to_show),
        )

    return preview_data


def save_import_data(request, cache_key):
    grid = _prepare_grid(request)
    previous_structure = grid.access_structure.copy()

    inserted_rows, _, changed_structure = do_import(request, cache_key, grid, False)

    save_imported_grid(request, grid, inserted_rows, previous_structure, changed_structure, grid.pk is not None)

    return grid


def get_parse_preview_data(request, cache_key):
    name, contents = _get_contents_from_cache(request, cache_key)
    grid_reader = _get_file_reader(name)(contents, **_get_format_params(request))
    grid_reader.open()

    preview_data = ParseDataPreview(cache_key, column_count=grid_reader.column_count, row_count=grid_reader.row_count)

    for row in grid_reader:
        preview_data.rows.append(row)
        if len(preview_data.rows) >= MAX_LINES_TO_PARSE_PREVIEW:
            break

    return preview_data


def validate_import_data(request, cache_key):
    grid_before_import = _get_grid(request)
    rows_before_import = len(grid_before_import.access_data) if grid_before_import else 0

    name, contents = _get_contents_from_cache(request, cache_key)
    grid_reader = _get_file_reader(name)(contents, **_get_format_params(request))
    grid_reader.open()

    limit = settings.LIMIT__GRID_ROWS_COUNT
    if limit and (rows_before_import + grid_reader.row_count) > limit:
        raise InvalidImportDataError(detail=ugettext('Maximum grid rows count exceeded'))

    limit = settings.LIMIT__GRID_COLS_COUNT
    if limit and grid_reader.column_count > limit:
        raise InvalidImportDataError(detail=ugettext('Maximum grid columns count exceeded'))

    limit = settings.LIMIT__WIKI_TEXT_FOR_GRID_CELL__BYTES
    if not limit:
        return

    for row in grid_reader:
        for cell in row:
            if len(cell.encode('utf-8')) > limit:
                raise InvalidImportDataError(
                    detail=ugettext('Maximum page size {limit} Kb exceeded').format(limit=int(limit / (1 << 10)))
                )
