from functools import cmp_to_key

from wiki.grids.models import Grid
from wiki.notifications import logic as notifications
from wiki.org import get_org
from wiki.pages.logic import subscription
from wiki.pages.models import Revision
from wiki.utils import cmp, timezone
from wiki.utils.supertag import tag_to_supertag
from wiki.utils.timer import track_time


def clone(grid, user, destination_tag, with_data):
    destination_supertag = tag_to_supertag(destination_tag)
    cloned_grid = Grid(
        modified_at=timezone.now(),
        tag=destination_tag,
        supertag=destination_supertag,
        owner=user,
        last_author=user,
        title=grid.title,
        page_type=Grid.TYPES.GRID,
        formatter_version=grid.formatter_version,
        mds_storage_id=grid.mds_storage_id.name,
        org=get_org(),
    )

    if not with_data:
        cloned_grid.remove(rows=list(grid.access_idx.keys()))

    cloned_grid.save()
    cloned_grid.authors.add(user)
    subscription.inherit_watches(page=cloned_grid, exclude_users=[user])
    subscription.create_watch(page=cloned_grid, user=user, is_cluster=False)

    notifications.create_page_event_create(
        page=cloned_grid,
        user=user,
        revision=Revision.objects.create_from_page(cloned_grid),
    )

    return cloned_grid


@track_time
def sort_grid_rows(rows, sorting):
    """
    @type rows: list
    """

    def get_sort_value(row, field_name):
        field = row.get(field_name) or {}
        value = field.get('sort')
        if value is None:
            value = field.get('raw')
        if isinstance(value, str):
            return value.lower()
        if value is None:
            return 0
        return value

    def sort(row1, row2):
        for sort_field in sorting:
            field_name = sort_field['name']
            res = cmp(get_sort_value(row1, field_name), get_sort_value(row2, field_name))
            if res:
                if sort_field['type'] == 'desc':
                    return -res
                return res
        return 0

    rows.sort(key=cmp_to_key(sort))
    return rows
