

from collections import OrderedDict
from collections.abc import Mapping

from cached_property import cached_property


class GridWrapper(Mapping):
    """
    Обертка для body грида для удобной работы со строками, столбцами, ячейками

    Если покажется удобным можно сделать MutableMapping — тогда можно будет
    изменять и сериализовать в dict.
    """

    def __init__(self, grid):
        self.grid = grid
        super(GridWrapper, self).__init__()

    @cached_property
    def data(self):
        grid_data = self.grid['data']
        matrix = OrderedDict()

        for row in grid_data:
            row_cells = OrderedDict()
            row_key = row['__key__']
            matrix[row_key] = row_cells
            for column_key in self.columns_keys:
                cell_value = row[column_key]
                row_cells[column_key] = cell_value

        return matrix

    @cached_property
    def fields(self):
        return self.grid['structure']['fields']

    def get_field_by_key(self, key):
        return {info['name']: info for info in self.fields}.get(key)

    @cached_property
    def rows_keys(self):
        rows_indexes = self.grid['idx']
        rows_order = [key_ for order_, key_ in sorted((order, key) for key, order in list(rows_indexes.items()))]
        return rows_order

    @cached_property
    def columns_keys(self):
        return [field['name'] for field in self.fields]

    def __getitem__(self, item):
        if isinstance(item, tuple) and len(item) == 2:
            row_key, column_key = item
            return self.data[row_key][column_key]
        return self.data[item]

    def __iter__(self):
        return iter(self.data)

    def __len__(self):
        return len(self.rows_keys)
