
import logging
import sys
import time
import traceback
from datetime import datetime

from django.core.management.base import BaseCommand
from django.db import transaction

from wiki.grids.models import Grid, Page, Revision
from wiki.utils.backports.mds_compat import APIError
from wiki.utils.models import queryset_iterator

log = logging.getLogger('wiki.management_commands')


def all_grids(grid_class, pk_gt):
    """
    @param grid_class: Класс возвращаемых значений - Grid либо Revision
    @type grid_class: Grid | Revision

    @param pk_gt: Нижняя граница (невключительно) primary key возвращаемых гридов
    @type pk_gt: int

    @return: Список всех гридов
    @rtype: list
    """
    grids_qs = grid_class.objects.all()

    # Чтобы возвращать ревизии только гридов, нужно
    # проверить тип соответствующей ревизии страницы.
    if grid_class == Revision:
        grids_qs = grids_qs.filter(page__page_type=Page.TYPES.GRID)

    for grid in queryset_iterator(grids_qs, pk_gt=pk_gt or 0):
        yield grid


DEFAULTS_FOR_EMPTY = {
    'string': '',
    'checkbox': False,
    'select_and_staff': [],
}


def process_grid_safe(grid):
    """
    @type grid: Grid
    """
    while True:
        try:
            process_grid(grid)

        # Таймаут хранилища, подождем, когда оно очнется
        except APIError as api_err:
            if 'read timed out' in api_err.message.lower() or 'timeout' in api_err.message.lower():
                print('[{now}] Storage timeout, retry in 5 seconds...'.format(now=datetime.now()))
                time.sleep(5)
                continue
            else:
                print('{exc_name}: {grid}'.format(exc_name=api_err.__class__.__name__, grid=grid))

        # Ошибки внутреннего формата гридов
        except (ValueError, TypeError) as e:
            print('{exc_name}: {grid}'.format(exc_name=e.__class__.__name__, grid=grid))

        # Память закончилась, деваться некуда, надо перезапускать миграцию
        except MemoryError:
            raise

        # Непредвиденное исключение, миграцию не останавливаем
        except Exception as e:
            print('Unexpected exception on {grid}: {message}'.format(grid=grid, message=e.message))
            traceback.print_exc(file=sys.stdout)

        break


@transaction.atomic
def process_grid(grid):
    """
    @type grid: Grid
    """
    # Заблокируем грид на чтение на время транзакции с помощью SELECT FOR UPDATE
    grid = grid.__class__.objects.select_for_update().get(pk=grid.pk)

    string_columns = list(grid.column_names_by_type('string'))
    checkbox_columns = list(grid.column_names_by_type('checkbox'))
    selects = list(grid.column_names_by_type('select', 'staff'))

    all_columns = list(grid.column_names_by_type())

    grid_is_dirty = False

    access_data = grid.access_data
    for row in access_data:
        for key in row:
            # проставим ячейкам хоть что-нибудь
            if row[key] is None:
                grid_is_dirty = True
                row[key] = {
                    'raw': None,
                }
        for column in all_columns:
            if column not in row:
                grid_is_dirty = True
                row[column] = {
                    'raw': None,
                }
        for column in string_columns:
            if isinstance(row[column], dict):
                if not isinstance(row[column].get('raw'), str):
                    grid_is_dirty = True
                    row[column]['raw'] = DEFAULTS_FOR_EMPTY['string']
            else:
                grid_is_dirty = True
                row[column] = {'raw': DEFAULTS_FOR_EMPTY['string']}
        for column in checkbox_columns:
            if isinstance(row[column], bool):
                grid_is_dirty = True
                row[column] = {'raw': row[column]}
            elif isinstance(row[column], dict):
                if not isinstance(row[column].get('raw'), bool):
                    grid_is_dirty = True
                    row[column]['raw'] = DEFAULTS_FOR_EMPTY['checkbox']
            else:
                grid_is_dirty = True
                row[column] = {'raw': DEFAULTS_FOR_EMPTY['checkbox']}
        for column in selects:
            if isinstance(row[column], dict):
                if not isinstance(row[column].get('raw'), list):
                    grid_is_dirty = True
                    row[column]['raw'] = DEFAULTS_FOR_EMPTY['select_and_staff']
            else:
                grid_is_dirty = True
                row[column] = {'raw': DEFAULTS_FOR_EMPTY['select_and_staff']}
        for column in all_columns:
            if not isinstance(row[column], dict):
                grid_is_dirty = True
                row[column] = {'raw': None}
    if grid_is_dirty:
        print('processing {}'.format(grid))
        delete_this_storage_id = grid.mds_storage_id.name
        grid.access_data = access_data
        grid.save()
        if isinstance(grid, Revision):
            # mds_storage_id у грида и его последней ревизии совпадают.
            # В миграции сначала обрабатываются гриды, затем ревизии.
            # Поэтому записи из хранилища удаляются только для ревизий,
            # иначе, если грид удалит запись из хранилища для своего mds_storage_id,
            # мы не сможем прочитать данные для его последней ревизии,
            # поскольку у нее такой же mds_storage_id. Вместе с тем, последняя ревизия
            # сама удалит эту запись из хранилища.
            grid.mds_storage_id.storage.delete(delete_this_storage_id)


class Command(BaseCommand):
    help = """Set strict values in grid fields"""

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--pk_gt',
            '-p',
            action='store',
            dest='pk_gt',
            default=None,
            help='Migrate grids with primary key greater than this parameter',
        )
        parser.add_argument(
            '--revisions_only',
            '-r',
            action='store_true',
            dest='revisions_only',
            default=False,
            help='Migrate only grid\'s revisions',
        )

    def handle(self, *args, **options):
        # Миграция долгая и по неизвестным причинам утекает по памяти.
        # Ввиду этого мы перехватываем MemoryError и пишем в stdout, как
        # запускать следующую миграцию. В случае непредвиденных исключений
        # мы поступаем так же.
        #
        # Миграция принимает два параметра -
        # --pk_gt - нижняя граница (невключительно) primary key мигрируемых гридов, и
        # --revisions_only - если параметр задан, то нужно мигрировать только
        # ревизии гридов (сами гриды мы уже промигрировали).
        #
        # Если --revisions_only не задан, то pk_gt касается primary key гридов,
        # если --revisions_only задан,    то pk_gt касается primary key ревизий.

        pk_gt = options.get('pk_gt')
        revisions_only = options.get('revisions_only')

        # Эти параметры нужны в случае падения миграции, чтобы
        # отобразить в stdout, как запускать миграцию в следующий раз.
        next_pk_gt = pk_gt
        next_revisions_only = revisions_only

        try:
            if not revisions_only:
                for grid in all_grids(Grid, pk_gt):
                    process_grid_safe(grid)
                    next_pk_gt = grid.pk

                # Гриды закончились, нужно сбросить параметр --pk_gt,
                # поскольку мы начинаем мигрировать ревизии с самого начала.
                pk_gt = None
                next_pk_gt = None
                next_revisions_only = True

            for grid in all_grids(Revision, pk_gt):
                process_grid_safe(grid)
                next_pk_gt = grid.pk

            print('Finished at {}'.format(datetime.now()))

        except Exception as e:

            print('[{now}] Unexpected exception: {message}'.format(now=datetime.now(), message=e.message))
            traceback.print_exc(file=sys.stdout)

            print(
                'Run "wiki-wiki-manage strict_values_in_grids{next_pk_gt}{next_revisions_only}"'.format(
                    next_pk_gt=' -p {}'.format(next_pk_gt) if next_pk_gt else '',
                    next_revisions_only=' -r' if next_revisions_only else '',
                )
            )
