
import logging
from django.core.management.base import BaseCommand
from functools import partial

from wiki.legacy.slave import is_slave

from wiki.utils import lock
from wiki.grids.models import Grid


log = logging.getLogger('management_commands.sync_grids')


class Command(BaseCommand):
    help = """Go through grids and sync their data
Updates staff if user has changed surname etc
"""
    requires_system_checks = False

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--simulate', '-s', action='store_true', dest='simulate', default=False, help='Dont save data'
        )

    def handle(self, *args, **options):
        if is_slave():
            return
        verbosity = int(options.get('verbosity'))
        simulate = options.get('simulate')
        if verbosity > 0 and simulate:
            print('Simulation mode')
        try:
            lock.execute_with_lock('sync_grids', partial(sync_grids, verbosity, simulate))
        except lock.FailedToGetLockError:
            pass
        except Exception as e:
            log.exception('Unhandled error at sync_grids command "%s"', repr(e))


def sync_grids(verbosity, simulate):
    grids = Grid.active.values_list('id', flat=True)
    for grid_id in grids:
        try:
            grid = Grid.objects.get(id=grid_id)
        except Grid.DoesNotExist:
            return
        if verbosity > 1:
            print(
                'Grid:pk={0},supertag="{1}"'.format(
                    grid.id,
                    grid.supertag,
                )
            )
        grid.sync_data()
        if not simulate:
            grid.save()
