
import functools
import logging
import re
import sys
from datetime import date

from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand
from django.db import transaction

from wiki.grids.models import Grid
from wiki.grids.utils import RowValidationError, insert_rows
from wiki.legacy.plocker import PLocker
from wiki.pages.api import prepare_request, save_page
from wiki.pages.models import Page
from wiki.utils.backports.mds_compat import APIError

logger = logging.getLogger('wiki.management_commands')


class FormatterErrorWrapper(Exception):
    def __init__(self, original_exception, trace):
        self.original_exception = original_exception
        self.trace = trace


def author_or_changes(page, default_author):
    return page.last_author or default_author


author_or_changes = functools.partial(
    author_or_changes, default_author=get_user_model().objects.all()[0]
)  # берем произвольного пользователя


def name_for_grid(prefix):
    """ Генерирует имя для нового грида, проверяя, что оно не занято """
    cnt = 1
    while True:
        while True:
            grid_name = '{0}/gridtable{1}'.format(prefix, cnt)
            cnt += 1
            try:
                Page.objects.get(supertag=grid_name)
            except Page.DoesNotExist:
                break
        yield grid_name


def prepare_grid_for(page):
    """Генерирует табличный список для заполнения данными"""
    for name in name_for_grid(page.supertag):
        yield Grid(supertag=name, tag=name, owner=author_or_changes(page), page_type=Page.TYPES.GRID)


def grid_structure(grid, columns, titles):
    fields = []
    for i in range(len(columns)):
        fields.append(
            {
                'name': str(i),  # обязательно так
                'type': columns[i],
                'required': False,
                'title': titles[str(i)] if titles else '',
            }
        )
    structure = {
        'title': 'auto generated grid',
        'fields': fields,
        'width': '100%',
        'sorting': [],
    }
    grid.change_structure(None, already_parsed=structure)
    return grid


def process_table_row(wom_node, body):
    """ Проходится по узлу wom-дерева "row" и выбирает значения внутри ячеек таблицы """
    result = {}
    hash = 0
    for cell in wom_node.get_children():
        children = cell.get_children()
        if children:
            result[str(hash)] = body[children[0].pos_start : children[-1].pos_end]
        else:
            result[str(hash)] = ''
        hash += 1
    return result


NUMBER_REGEX = re.compile(r'\d+((?:\.|,)\d+)?$')
DATE_REGEX = (
    re.compile(r'(?P<year>\d{4})-(?P<month>\d{2})-(?P<day>\d{2})'),  # YYYY-MM-DD ISO 8601
    # DD.MM.YYYY русский, межд. английский DD-MM-YYYY
    re.compile(r'(?P<day>\d{2})(?:\.|-)(?P<month>\d{2})(?:\.|-)(?P<year>\d{4})'),
    re.compile(r'(?P<month>\d{2})/(?P<day>\d{2})/(?P<year>\d{4})'),  # MM/DD/YYYY английский
)


def guess_datatypes(row):
    """По данному списку строк пытается догадаться об их типах и приводит строки к этим типам

    Умеет работать с типами "число", "дата", "строка".

    >>> guess_datatypes(["10,0", "1.25"])
    ((10.0, 1.25), ["number", "number"))
    >>> guess_datatypes(["1963-01-20", "10/20/1964", "20/20/1984"]) # third string is not a date
    (("1963-01-20", "1964-10-20", "20/20/1984"), ("date", "date", "string"))
    >>> guess_datatypes([""])
    (('',), ("any",))
    """
    types = []

    for i in range(len(row)):
        key = str(i)  # так мы сохраняем порядок в types
        cell = row[key].strip()  # убираем пробельные символы
        if not cell:  # пустое значение соответствует виртуальному типу "any"
            row[key] = ''
            types.append('any')
            continue
        if NUMBER_REGEX.match(cell):  # число
            row[key] = cell.strip().replace(',', '.')
            types.append('number')
            continue
        curr_type = 'string'
        for regex in DATE_REGEX:
            match = regex.match(cell)
            if not match:
                continue
            data = match.groupdict()
            try:
                date(int(data['year']), int(data['month']), int(data['day']))
            except ValueError:  # это похоже на дату, но не дата
                break
            curr_type = 'date'
            row[key] = '{year}-{month}-{day}'.format(**data)
        types.append(curr_type)
    return row, tuple(types)


def multiply_datatypes(types1, types2):
    """Любые несовпадения приводятся к строке.

    Это ненастоящее умножение. Но похоже, string это 0, any это 1
    """
    result = []
    for i in range(len(types1)):
        if 'any' == types1[i]:
            result.append(types2[i])
        elif 'any' == types2[i]:
            result.append(types1[i])
        elif types1[i] != types2[i]:
            result.append('string')
        else:
            result.append(types1[i])
    return tuple(result)


def postprocess_datatypes(column_type):
    """ Вспомогательная функция, которая убирает виртуальный тип "any" """
    if column_type == 'any':
        return 'string'
    return column_type


def strip_table_head(row):
    """ Хелпер, чтобы убрать символы "**" из разметки заголовка грида """
    for key in row:
        if row[key]:
            row[key] = row[key].strip().strip('**')
    return row


def recognize_table_head(row, row_types, table_types):
    """Точно отвечает, является ли строка заголовком для таблицы

    1. Если некоторые типы данных в таблице не типа "строка",
    а в первой строке все типы "строка", то такая строка -- заголовок.

    2. Если все ячейки в первой строке обернуты маркером полужирного текста
    """
    if any(_type != 'string' for _type in table_types) and all(_type == 'string' for _type in row_types):  # условие 1
        return strip_table_head(row)

    is_wrapped = lambda s: s.strip()[:2] == '**' and s.strip()[-2:] == '**'
    if all(is_wrapped(cell) for cell in list(row.values())):  # условие 2
        return strip_table_head(row)

    return None


def correct_page_body(body, root, grid_gen, page):
    cnt = 0
    for wom_node in reversed(list(root.filter('table', type=0))):  # таблицы с type=0 это таблицы с рамками
        save = True
        rows = []
        column_types = None
        wom_rows = wom_node.filter('row')
        if len(wom_rows) < 2:  # не преобразуем таблицы из менее чем двух строк
            continue
        first_row = wom_rows[0]  # первая строка может быть заголовком
        wom_rows = wom_rows[1:]
        for child in wom_rows:
            row, types = guess_datatypes(process_table_row(child, body))
            if column_types is None:
                column_types = types
                cnt += 1
            else:
                if len(column_types) != len(types):  # число столбцов в рядочках
                    save = False  # разное, поэтому пропускаем эту таблицу
                    break
                if column_types != types:  # типы данных не совпадают
                    column_types = multiply_datatypes(column_types, types)
            rows.append(row)

        row, types = guess_datatypes(process_table_row(first_row, body))  # пытаемся извлечь названия столбцов грида
        if len(column_types) != len(types):
            save = False
        column_titles = recognize_table_head(row, types, column_types)
        if not column_titles:
            rows.insert(0, row)
        if rows and save:  # не создаем пустые таблицы
            grid = next(grid_gen)  # создаем новый табличный список
            grid_structure(grid, list(map(postprocess_datatypes, column_types)), column_titles)
            request = prepare_request(page.supertag, author_or_changes(page))
            insert_rows(grid, rows, request=request)
            grid.save()
            body = (
                body[: wom_node.pos_start]
                + "{{{{grid page=\"/{0}\"}}}}".format(grid.supertag)
                + body[wom_node.pos_end :]
            )
    return body


class Command(BaseCommand, PLocker):
    help = """Change all tables on page into grids"""

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--simulate', '-s', actions='store_true', dest='simulate', default=False, help='Don\'t save data'
        )
        parser.add_argument('--all', '-a', action='store_true', dest='all', default=False, help='Process all grids')

    def handle(self, *args, **options):
        if self.is_slave():
            print('Server is readonly')
            return
        verbosity = int(options.get('verbosity'))
        simulate = options.get('simulate')
        if verbosity > 0 and simulate:
            print('Dry-run mode')
        if not (args or options.get('all')):
            if verbosity > 0:
                print('Nothing to do', file=sys.stderr)
            return
        qs = (
            Page.active.values_list('supertag')
            .exclude(page_type=Page.TYPES.GRID)  # выбираем неудаленные страницы,
            .filter(redirects_to__isnull=True)  # которые не табличные списки
        )  # и не редиректы
        if not options.get('all'):
            qs = qs.filter(supertag__in=args)

        for supertag in (value[0] for value in qs):
            self.do_process(supertag, verbosity, simulate)

    def process(self, supertag, verbosity, simulate):
        transaction.set_autocommit(False)
        if self.do_process(supertag, verbosity, simulate):
            transaction.commit()
        else:
            transaction.rollback()

    def do_process(self, supertag, verbosity, simulate):
        try:
            page = Page.active.get(supertag=supertag)
        except Page.DoesNotExist:
            return
        if verbosity > 1:
            print("Processing page \"{0}\"".format(page.id))
        try:
            root = page.wom0_root
        except APIError:  # ошибка в хранилище
            print('Can\'t get storage body for page "{0}"'.format(page.id), file=sys.stderr)
            return
        except FormatterErrorWrapper as e:
            print("formatter error on page \"{0}\"".format(page.supertag), file=sys.stderr)
            if simulate or verbosity > 1:
                print(e.trace, file=sys.stderr)
            return
        grid_generator = prepare_grid_for(page)
        try:
            new_body = correct_page_body(page.body, root, grid_generator)
        except RowValidationError as e:
            print("error inserting rows into grid \"{0}\"".format(e.errors), file=sys.stderr)
            return
        except AttributeError:  # например, CSV-форматтер создает таблицу, у которой нет pos_start
            print("error parsing table on page \"{0}\"".format(page.id), file=sys.stderr)
            return

        if new_body != page.body:
            if verbosity > 0:
                print("Changed page \"{0}\"".format(page.id))
            if not simulate:
                author = author_or_changes(page)
                save_page(page, new_body, title=page.title, user=author, authors=author)
