from django.db import models
from wiki.utils import timezone

from wiki.pages.models import Page
from wiki.grids.utils.grid_interface import GridInterfaceMixin


class GridManager(models.Manager):
    def get_queryset(self):
        return super(GridManager, self).get_queryset().filter(page_type=Grid.TYPES.GRID)


class GridActiveManager(GridManager):
    def get_queryset(self):
        return super(GridActiveManager, self).get_queryset().filter(status__gt=0)


class Grid(GridInterfaceMixin, Page):
    class Meta:
        proxy = True

    objects = GridManager()
    active = GridActiveManager()

    def save(self, *args, **kwargs):
        if not self.modified_at:
            self.modified_at = timezone.now()
        super(Grid, self).save(*args, **kwargs)

    def get_document(self):
        elements = [
            str(row[key].get('raw', '')) if isinstance(row[key], dict) else ''
            for row in self.access_data
            for key in row
        ]
        return {'title': self.title, 'body': '\n'.join(elements), 'keywords': self.get_keywords()}

    def get_metadata(self):
        metadata = Page.get_metadata(self)
        metadata['type'] = 'grid'
        return metadata
