import logging
import time
from copy import deepcopy

from django.conf import settings
from django.template.defaultfilters import escape
from django.utils.encoding import smart_text
from django.utils.safestring import SafeData
from django.utils.translation import ugettext, ugettext_lazy

from wiki.grids.logic.tickets import ticket_field_names
from wiki.legacy.choices import Choices
from wiki.org import org_staff
from wiki.users.logic.settings import get_user_lang_ui

ticket_statuses = Choices(
    NEW='new',
    FAILED='failed',
    PRIVATE='private',
    NORMAL='normal',
)

HASH_KEY = '__key__'
COLOR_KEY = '__has_color__'

logger = logging.getLogger(__name__)

STAFF_TYPE = 'staff'
SELECT_TYPE = 'select'
CHECKBOX_TYPE = 'checkbox'

native_field_types = {
    'string': ugettext_lazy('grids.field:String'),
    SELECT_TYPE: ugettext_lazy('grids.field:Select'),
    'number': ugettext_lazy('grids.field:Number'),
    STAFF_TYPE: ugettext_lazy('grids.field:Staff'),
    'date': ugettext_lazy('grids.field:Date'),
    CHECKBOX_TYPE: ugettext_lazy('grids.field:Checkbox'),
}

the_field_types = native_field_types.copy()
TICKET_TYPE = 'ticket'

if settings.ISSUE_TRACKERS_ENABLED:
    the_field_types[TICKET_TYPE] = ugettext_lazy('grids.field:Ticket')


def serialized_field(field_name):
    return '__unserialized_' + field_name


class FieldValidationError(Exception):
    message = ugettext('validation error')

    def __init__(self, field_name, *args, **kwargs):
        self.field_name = field_name
        Exception.__init__(self, *args, **kwargs)


class DateFormatInvalid(FieldValidationError):
    message = ugettext('Enter a valid date, e.g. 2011/07/06')


class NumberFormatInvalid(FieldValidationError):
    message = ugettext('Enter a valid number')


class DubiousFieldName(FieldValidationError):
    message = ugettext('grids:Dubious field name')


def sanitize_value(value):
    if isinstance(value, list):
        return list(map(escape, value))
    elif isinstance(value, SafeData):
        return value
    elif isinstance(value, str):
        return escape(value)
    return value


def sanitize_structure(struct):
    if isinstance(struct, list):
        for i in range(len(struct)):
            struct[i] = sanitize_structure(struct[i])
    elif isinstance(struct, dict):
        keys = list(struct.keys())
        for key in keys:
            sane = sanitize_structure(key)
            if key != sane:
                struct[sane] = struct[key]
                del struct[key]
        for key in struct.keys():
            struct[key] = sanitize_structure(struct[key])
    else:
        struct = sanitize_value(struct)
    return struct


def dummy_request_for_grids(user=None):
    """Вернуть объект, похожий на HttpRequest.

    Объект подходит для тестирования и отладки текстовых полей в гридах

    @return: object
    """
    from django.contrib.auth.models import AnonymousUser
    from django.http import HttpRequest

    request = HttpRequest()
    setattr(request, 'user', user or AnonymousUser())
    setattr(request, 'page', None)
    setattr(request, 'LANGUAGE_CODE', get_user_lang_ui(user))
    setattr(request, 'user_auth', None)

    return request


def expand_date_field(data, name, **kwargs):
    input_value = kwargs.get('input_value')
    try:
        if data[name].strip():
            time.strptime(input_value, '%Y-%m-%d')
        return {'raw': data[name]}
    except ValueError:
        raise DateFormatInvalid(name)


def expand_staff_field(data, name, **kwargs):
    input_value = kwargs.get('input_value')
    description = kwargs.get('description')

    if not input_value:
        value = {'raw': []}
    elif isinstance(input_value, list):
        # Отфильтровываем пустые значения
        input_value = list(filter(None, input_value))
        value = {'raw': input_value}
    else:
        value = {'raw': [val.strip() for val in input_value.strip().split(',')]}
    kwargs.get('staff_syncer').sync_raw(
        value,
        description.get('format', 'i_first_name i_last_name'),
        multiple=description.get('multiple', False),
    )
    return value


def expand_number_field(data, name, **kwargs):
    input_value = kwargs.get('input_value')
    if input_value:
        possible_number = data[name]
        if isinstance(possible_number, float) or isinstance(possible_number, int) or isinstance(possible_number, int):
            return {'raw': possible_number, 'sort': possible_number}
        elif isinstance(possible_number, str):
            possible_number = possible_number.replace(',', '.')
        try:
            int_value = int(possible_number)
            return {'raw': int_value, 'sort': int_value}
        except (TypeError, ValueError):
            try:
                float_value = float(possible_number)
                return {'raw': float_value, 'sort': float_value}
            except (TypeError, ValueError):
                raise NumberFormatInvalid(name)
    else:
        return {'raw': None}


def expand_select_field(data, name, **kwargs):
    input_value = kwargs.get('input_value')
    if isinstance(input_value, list):
        return {'raw': list(input_value)}
    else:
        return {'raw': [input_value] if input_value else list()}


def expand_checkbox_field(data, name, **kwargs):
    input_value = kwargs.get('input_value')
    if input_value == '':
        return {'raw': False}
    else:
        return {'raw': True}


def expand_string_field(data, name, **kwargs):
    return {'raw': smart_text(data[name])}


def tracker_issue_link(issue_key):
    return '<span class="jiraissue"><a href="%s/%s">%s</a></span>' % (
        settings.STARTREK_CONFIG['INTERFACE_URL'],
        issue_key,
        issue_key.upper(),
    )


def expand_ticket_field(data, name, **kwargs):
    input_value = kwargs.get('input_value')
    if input_value:
        html = tracker_issue_link(input_value)
    else:
        html = ''
    return {
        'raw': input_value.upper(),
        'view': html,
    }


def expand_default(*args, **kwargs):
    return {'raw': kwargs.get('input_value')}


_expand_type_map = {
    'date': expand_date_field,
    'staff': expand_staff_field,
    ticket_field_names.assignee: expand_staff_field,
    ticket_field_names.reporter: expand_staff_field,
    'number': expand_number_field,
    'select': expand_select_field,
    'string': expand_string_field,
    'ticket': expand_ticket_field,
    'checkbox': expand_checkbox_field,
}


def expand_type_map(type):
    return _expand_type_map.get(type, expand_default)


def filter_data_from_client(grid, data, request):
    """Провалидировать данные от клиента.

    Дополняет табличный список значениями "" для необязательных полей.
    raises DateFormatInvalid and NumberFormatInvalid

    http://wiki.yandex-team.ru/WackoWiki/dev/grid#formatpriemaperedachidanny
    """
    if request is None:
        request = dummy_request_for_grids()
    result = {}
    structure = grid.access_structure
    staff_syncer = StaffSyncer()
    for name in data:
        if grid.column_by_name(name) is not None:
            continue
        raise DubiousFieldName(name)
    for description in structure['fields']:
        name = description['name']
        if name not in data:
            continue
        expander = expand_type_map(description['type'])
        result[name] = expander(
            data,
            name,
            **{
                'input_value': sanitize_value(data[name]),
                'staff_syncer': staff_syncer,
                'description': description,
                'grid': grid,
                'request': request,
            }
        )
    return result


class ModelCache:
    """
    Remembers objects, that you have asked it, and returns them if found in
    storage.
    Do not use it if you use single object once. It has no meaning.
    """

    def __init__(self, query_set):
        self.qs = query_set
        self.storage = set([])

    def _do_get(self, field_name, values):
        """get directly from storage without updating it"""
        return [_ for _ in self.storage if getattr(_, field_name) in values]

    def get(self, field_name, values):
        values = [v for v in values if v]  # не позволяю пустых полей, типа ['']
        if not values:
            return []
        result = self._do_get(field_name, values)
        if len(result) == len(values):
            return result
            # probably not enough items in storage
        for entity in self.qs.filter(**{field_name + '__in': values}):
            self.storage.add(entity)
        return self._do_get(field_name, values)


class StaffSyncer(object):
    """updates the fields of the given grid with staff data type"""

    cache = None
    ALLOWED_STAFF_ATTRS = [
        'login',
        'domain',
        'first_name',
        'middle_name',
        'last_name',
        'en_name',
        'birthday',
        'family_status',
        'children',
        'car',
        'car_num',
        'address',
        'edu_place',
        'edu_date',
        'office',
        'join_at',
        'desk_id',
        'quit_at',
        'employment',
        'work_phone',
        'mobile_phone',
        'all_groups',
        'email',
        'department',
        'gender',
        'i_first_name',
        'i_last_name',
    ]

    def sync(self, grid, hash=None):
        """
        Update data and return grid if something was updated.

        @type grid: Grid
        """
        fields_to_sync = [
            (field['name'], field['format'], field['multiple']) for field in grid.columns_by_type('staff')
        ]
        if not fields_to_sync:
            return None
        all_data = list(grid.access_data)
        has_been_modified = False
        for row in all_data:
            for field in fields_to_sync:
                data = row.get(field[0], False)
                if data is False or data is None:
                    logger.info(
                        "The row has no staff data for field '%s' grid.supertag '%s', row %s",
                        field,
                        grid.supertag,
                        repr(row),
                    )
                    continue
                if data.get('raw', False) is False:
                    logger.warning(
                        "The row has no raw data for field '%s' grid.supertag '%s', row %s",
                        field,
                        grid.supertag,
                        repr(row),
                    )
                    continue
                if self.sync_raw(data, field[1], multiple=field[2]):
                    has_been_modified = True
        grid.access_data = all_data
        return grid if has_been_modified else None

    def get_attrs(self, staff, attrs):
        result = []
        for attr in attrs.split(' '):
            if attr in self.ALLOWED_STAFF_ATTRS:
                result.append(getattr(staff, attr))
            else:
                result.append(attr)
        return ' '.join(result)

    def sync_raw(self, data, attrs, multiple, **kwargs):
        """
        Обновить поле типа staff, вернуть False если поле не поменялось.

        @type data: dict
        @type attrs: str
        @type multiple: bool
        @rtype: bool
        """
        if self.cache is None:
            self.cache = ModelCache(org_staff())
        transformed = {}
        sort = ''
        unmodified_data = deepcopy(data)
        raw = data['raw']
        if not hasattr(raw, '__iter__') or isinstance(raw, str):
            raw = [raw] if raw else list()
            data['raw'] = raw
        staff = self.cache.get('login', raw)
        for login in raw:
            person = [_staff for _staff in staff if _staff.login == login]
            if len(person) != 1:
                transformed[login] = login
                if not multiple:
                    sort = login
                continue
            transformed[login] = self.get_attrs(person[0], attrs)
            if not multiple:
                sort = self.get_attrs(person[0], 'i_last_name i_first_name')

        data['transformed'] = transformed
        if not multiple:
            data['sort'] = sort

        return data != unmodified_data


def insert_missing_fields(grid, row_count=0):
    """
    Decorator for inserting full row, when user supplied only a few values
    """
    fields = set(field['name'] for field in grid.columns_meta())

    def do_insert(row):
        for field_name in fields:
            if field_name in row:
                continue
            row[field_name] = ''
        return row

    def wrap(iter):
        cnt = 0
        for row in iter:
            cnt += 1
            yield do_insert(row)
            if row_count and cnt == row_count:
                break

    return wrap
