from collections import defaultdict
from json import loads

from wiki.grids.utils.view import grid_type_to_td, plain_value
from wiki.notifications.generators.base import EventTypes
from wiki.pages.models import Revision


def compare_rows(row1, row2, grid):
    columns = list(grid.access_structure['fields'])
    column_descriptions = dict([(_['name'], _) for _ in columns])
    names = [_['name'] for _ in columns]
    plain_row1, plain_row2 = (
        [plain_value(grid_type_to_td(row[key], column_descriptions[key])) for key in names] for row in (row1, row2)
    )
    different = []
    for i in range(len(plain_row1)):
        if plain_row1[i] != plain_row2[i]:
            different.append(names[i])
    return different


RESOLUTIONS = 'resolutions'


def compress_data(page_event, **kwargs):
    """Compress PageEvent with changes in data
    @param page_event PageEvent
    """
    meta = page_event.meta.copy()
    meta[RESOLUTIONS] = defaultdict(list)
    data = meta['data']
    for id in data:
        if len(data[id]) == 0:
            continue
        was_deleted = False
        was_added = False
        for event in reversed(data[id]):
            if event['type'] == EventTypes.delete:
                was_deleted = event
            if event['type'] == EventTypes.create:
                was_added = event
        if was_added and was_deleted:
            continue
        elif was_deleted:
            if 'old_row' in data[id][0]:  # to remove double row
                del data[id][0]['old_row']
            meta[RESOLUTIONS][str(EventTypes.delete)].append(data[id][0])
        elif was_added:
            if 'old_row' in data[id][-1]:  # to remove double row
                del data[id][-1]['old_row']
            meta[RESOLUTIONS][str(EventTypes.create)].append(data[id][-1])
        else:
            edited_row = data[id][-1]
            if 'old_row' in data[id][0]:
                old_row = data[id][0]['old_row']
            else:
                old_row = data[id][0]['row']
            edited_row['old_row'] = old_row

            try:
                revision = Revision.objects.get(id=page_event.meta.get('revision_id'))
            except Revision.DoesNotExist:
                pass
            else:
                from wiki.grids.models import Grid

                grid = Grid()
                grid.change_structure(None, loads(revision.body)['structure'])
                edited_row['different_columns'] = compare_rows(old_row, edited_row['row'], grid)
            meta[RESOLUTIONS][str(EventTypes.edit)].append(edited_row)
    page_event.meta = meta


def compare_fields(previous, current):
    deleted, created, changed = [], [], []
    deleted_names, changed_names, untouched_names = [], [], []
    for field in previous:
        the_filter = [f for f in current if f['name'] == field['name']]
        was_deleted = False if the_filter else True
        if was_deleted:
            deleted_names.append(field['name'])
            deleted.append(field)
        else:
            if the_filter[0] != field:
                changed_names.append(field['name'])
                changed.append(field)
            else:
                untouched_names.append(field['name'])
    containers = (deleted_names, changed_names, untouched_names)
    for field in current:
        if any(field['name'] in container for container in containers):
            continue
        created.append(field)
    return {
        str(EventTypes.create): created,
        str(EventTypes.edit): changed,
        str(EventTypes.delete): deleted,
    }


def compare_order_of_fields(first_structure, current_structure):
    field_names = lambda struct: [f['name'] for f in struct]
    previous, current = field_names(first_structure.get('fields')), field_names(current_structure.get('fields'))
    if len(previous) != len(current):
        return False
    if set(previous) != set(current):
        return False
    for i in range(len(previous)):
        if previous[i] != current[i]:
            return True
    return False


def compare_sorting(first_structure, current_structure):
    previous, current = first_structure.get('sorting'), current_structure.get('sorting')
    if previous is None or current is None:
        return []
    if previous == current:
        return []
    sorting = []
    for sort in current:
        the_filter = [f for f in current_structure['fields'] if sort['name'] == f['name']]
        if not the_filter:
            continue
        field = the_filter[0]
        sorting.append({'name': field['title'] or field['name'], 'type': sort['type']})
    return sorting


def compress_structure(page_event, **kwargs):
    meta = page_event.meta.copy()
    first_structure = meta['structure'][0]['structure_before']
    current_structure = meta['structure'][-1]['structure']
    meta[RESOLUTIONS] = {
        'fields': compare_fields(first_structure['fields'], current_structure['fields']),
        'sorting': compare_sorting(first_structure, current_structure),
        'order': compare_order_of_fields(first_structure, current_structure),
    }

    page_event.meta = meta


strategy = {
    'data': compress_data,
    'structure': compress_structure,
}


def compress_single(page_event):
    caller = strategy.get(page_event.meta.get('object_of_changes'))
    if caller is not None:
        caller(page_event)
    if page_event.sent_at is not None:
        page_event.save()


def compress(query_set):
    """These functions add key "resolutions" key to meta.
    They also save the meta if information about it has been sent (sent_at is not None)
    Please do not save them manually if using these methods
    """

    for page_event in query_set:
        if page_event.event_type != EventTypes.edit:
            continue
        if RESOLUTIONS in page_event.meta:
            continue
        compress_single(page_event)
