
import io
import csv

import xlwt

from docx import Document

from wiki.grids.utils.base import STAFF_TYPE
from wiki.grids.logic.operations import sort_grid_rows

CSV_ENCODING = 'utf-8'
CSV_DIALECT = 'excel'


def export_xls(structure, data):
    """
    Экспорт из атрибутов грида access_structure и access_data в xls
    @rtype: BytesIO
    """

    headers, rows = _extract_rows(structure, data)

    workbook = xlwt.Workbook()
    worksheet = workbook.add_sheet('Sheet1')

    bold_font = xlwt.Font()
    bold_font.bold = True
    bold_style = xlwt.XFStyle()
    bold_style.font = bold_font

    for j, caption in enumerate(headers):
        worksheet.write(0, j, caption, bold_style)

    for i, row in enumerate(rows, start=1):
        for j, cell in enumerate(row):

            # Если данных нет, пропускаем
            if cell is None:
                continue

            if type(cell) is bool:
                cell = ('', '✓')[cell]

            elif type(cell) is list:
                # Для подстраховки привожу к unicode элементы списка, хотя они итак должны быть unicode.
                cell = ', '.join(str(c) for c in cell)

            elif type(cell) is not int:
                cell = str(cell)

            worksheet.write(i, j, cell)

    xls_file = io.BytesIO()
    workbook.save(xls_file)

    return xls_file


def export_docx(structure, data):
    """
    Экспорт из атрибутов грида access_structure и access_data в docx
    @rtype: BytesIO
    """

    doc = Document()

    add_grid_to_docx(structure, data, doc)

    docx_file = io.BytesIO()
    doc.save(docx_file)

    return docx_file


def add_grid_to_docx(structure, data, doc):
    headers, rows = _extract_rows(structure, data)

    table = doc.add_table(rows=len(rows) + 1, cols=len(headers))
    table.style = 'TableGrid'

    for header_index, header in enumerate(headers):
        table_cell = table.rows[0].cells[header_index]
        run = table_cell.paragraphs[-1].add_run(header)
        run.bold = True

    for i, row in enumerate(rows, start=1):
        for j, cell in enumerate(row):
            # Если данных нет, пропускаем
            if cell is None:
                continue

            if type(cell) is bool:
                cell = ('', '✓')[cell]

            elif type(cell) is list:
                # Для подстраховки привожу к unicode элементы списка, хотя они итак должны быть unicode.
                cell = ', '.join(str(c) for c in cell)

            elif type(cell) is not int:
                cell = str(cell)

            table_cell = table.rows[i].cells[j]
            table_cell.paragraphs[-1].add_run(cell)


def export_csv(structure, data):
    """
    Экспорт из атрибутов грида access_structure и access_data в csv
    @rtype: StringIO
    """

    headers, rows = _extract_rows(structure, data)

    csv_file = io.StringIO()
    csv_writer = csv.writer(csv_file, dialect=CSV_DIALECT)
    csv_writer.writerow(list(headers))
    for row in rows:
        formatted_row = []
        for cell in row:
            if cell is None:
                formatted_row.append('')
            elif type(cell) is list:
                formatted_row.append(', '.join(str(c) for c in cell))
            else:
                formatted_row.append(str(cell))
        csv_writer.writerow(list(formatted_row))

    return csv_file


def _extract_rows(structure, data):
    """
    Превращает атрибуты грида access_structure и access_data в список заголовков и список строк, пригодных для экспорта.
    """

    fields = structure['fields']
    headers = [f['title'] for f in fields]
    rows = []
    # отсортировать данные перед экспортом согласно гридовым настройкам сортировки
    data = sort_grid_rows(data, structure.get('sorting'))

    for row in data:
        row_data = []
        for field in fields:
            field_id = field['name']

            cell = row.get(field_id)
            cell_data = None if cell is None else cell.get('raw')

            # Добавляем в гридах в полях staff к логинам имена людей
            if cell_data and field['type'] == STAFF_TYPE and cell.get('transformed'):
                for i, login in enumerate(cell_data):
                    if login in cell['transformed']:
                        cell_data[i] = '%s (%s)' % (cell['transformed'][login], login)

            row_data.append(cell_data)

        rows.append(row_data)

    return headers, rows
