"""
TODO: унести функции из класса в wiki.grids.dao, оставить GridInterfaceMixin простой оберткой
"""
import logging
from json import dumps, loads

from wiki.utils import timezone
from django.utils.safestring import SafeText

from wiki.grids.utils import sanitize_structure
from wiki.grids.utils import StaffSyncer, unserialize_data, make_beautiful_structure_from, HASH_KEY, SerializeAsDictKey
from wiki.grids import logic as grid_logic
from wiki.grids.utils import ticket_field_names

SORTABLE_FIELD_TYPES = (
    'date',
    'string',
    'number',
    'select',
    'staff',
    'checkbox',
    'ticket',
    ticket_field_names.assignee,
    ticket_field_names.reporter,
    ticket_field_names.status,
    ticket_field_names.priority,
    ticket_field_names.type,
    ticket_field_names.subject,
    ticket_field_names.created_at,
    ticket_field_names.updated_at,
    ticket_field_names.original_estimation,
    ticket_field_names.fix_versions,
    ticket_field_names.components,
    ticket_field_names.story_points,
)

logger = logging.getLogger()


class GridInterfaceMixin(object):
    """Добавить к объектам похожим на гриды общий интерфейс доступа к данным."""

    access_structure = SerializeAsDictKey('structure', 'body', dumps, make_beautiful_structure_from)
    access_data = SerializeAsDictKey('data', 'body', dumps, make_beautiful_structure_from)
    access_idx = SerializeAsDictKey('idx', 'body', dumps, make_beautiful_structure_from)
    access_meta = SerializeAsDictKey('meta', 'body', dumps, make_beautiful_structure_from)

    def _rebuild_index_at(self, row_number=0):
        idx = self.access_idx.copy()
        data = self.access_data
        for i in range(row_number, len(data)):
            idx[data[i][HASH_KEY]] = i
        self.access_idx = idx

    def row_id_and_row(self):
        """
        Вернуть пары "id строки", "строка"
        """
        rows = getattr(self, 'deserialized_rows', self.access_data)
        for row in rows:
            yield row[HASH_KEY], row

    def get_rows(self, user_auth, hashes=None):
        """Обрабатывает строки перед отдачей клиенту.

        @param hashes: list of hashes
        """
        numbers = None
        # FIXME: отдавать лишь те столбцы, которые нужны, а не все подряд
        if hashes is None:
            numbers = range(len(self.access_data))

        result = unserialize_data(
            self.body,
            hashes=hashes,
            numbers=numbers,
            user_auth=user_auth,
        )
        return result

    def remove(self, rows):
        """
        rows is a list of HASHes
        """
        idx = self.access_idx
        data = self.access_data
        indexes_to_remove = []
        lowest_row = 32535
        deleted_data = {}
        for row in rows:
            # check if row in access_idx
            i_row = idx.get(row)
            if i_row is not None:
                if i_row < lowest_row:
                    lowest_row = i_row
                indexes_to_remove.append(i_row)
                deleted_data[row] = data[i_row]
                del idx[row]

        # delete from the highest to maintain right indexes
        indexes_to_remove.sort(reverse=True)
        for index in indexes_to_remove:
            data.pop(index)
        self.access_data = data
        self._rebuild_index_at(lowest_row)
        self.modified_at = timezone.now()
        return deleted_data

    def remove_column(self, column_name):
        """
        Удалить столбец.

        @type column_name: basestring
        @raises: ValueError
        """
        return grid_logic.remove_column(self, column_name)

    def move(self, rows, after=None):
        idx = self.access_idx
        if not after:
            idx_after = -1
        else:
            idx_after = idx[after]

        self.move_after_idx(rows, after_idx=idx_after)

    def move_after_idx(self, rows, after_idx):
        data = self.access_data
        rows.reverse()
        for row_id in rows:
            index = self.access_idx[row_id]
            row = data.pop(index)
            if index < after_idx:  # to compensate, that data became 1 element shorter
                after_idx -= 1
            data.insert(after_idx + 1, row)  # because method inserts BEFORE the given index
            self._rebuild_index_at(min(index, after_idx))
        self.access_data = data

    def sync_data(self, hash=None):
        syncer = StaffSyncer()
        return syncer.sync(self, hash)

    def columns_meta(self):
        """
        @rtype: list
        """
        return self.access_structure['fields']

    def columns(self):
        for index, column in enumerate(self.columns_meta()):
            yield index, column['type'], column

    def columns_by_type(self, *column_types):
        """
        Вернуть столбцы заданных типов.

        Если column_types не задано, то результатом работы будут все столбцы.

        @type grid: Grid
        @rtype: dict
        """
        return grid_logic.columns_by_type(self, *column_types)

    def column_by_name(self, column_name):
        """
        Вернуть столбец по имени или None.
        """
        return grid_logic.column_by_name(self, column_name)

    def column_names_by_type(self, *column_types):
        """
        Вернуть имена столбцов с заданным типом.

        Если column_types не задано, то результатом работы будут все столбцы.
        Порядок следования имен столбцов сохраняется.

        @rtype: list
        """
        return grid_logic.column_names_by_type(self, *column_types)

    def column_titles(self, *column_names):
        """
        Вернуть человекочитаемые названия столбцов грида по их именам.

        @rtype: dict
        """
        return dict(list(zip(column_names, grid_logic.column_attribute_by_names(self, 'title', *column_names))))

    def change(self, row_id, row_data):
        """This method is UNSAFE - it does not check "row_data" for consistency"""
        data = self.access_data
        data[self.access_idx[row_id]].update(row_data)
        self.access_data = data
        self.modified_at = timezone.now()

    def change_structure(self, plain_structure, already_parsed=None):
        """Обновить все строки с данными, добавив нужные значения

        Must update all data rows, removing all deleted fields,
        adding new fields, filled with empty values

        @params plain_structure: string, plain YAML
        @params already_parsed: python object
        """
        new_structure = already_parsed or loads(plain_structure)
        # Не санитайзим заголовок грида
        new_structure['title'] = SafeText(new_structure.get('title', ''))
        new_structure = sanitize_structure(new_structure)
        if not isinstance(new_structure, dict):
            raise TypeError('Please, give me a dict')
        structure = self.access_structure
        if new_structure != structure:
            if 'fields' not in structure:
                structure['fields'] = []
            old_fields = structure['fields']
            if 'fields' not in new_structure:
                new_structure['fields'] = []
            new_fields = new_structure['fields']
            new_field_names = [f['name'] for f in new_fields]
            if old_fields != new_fields:
                data = self.access_data
                old_field_names = [f['name'] for f in old_fields]
                fields_to_delete = []  # plain list
                fields_to_change = {}  # pairs: field name - value
                for field in new_fields:
                    update_fields(field, old_fields, old_field_names, fields_to_change)
                for field_name in old_field_names:
                    # delete fields which do no exist anymore
                    # do NOT touch fields which start with space
                    if not (field_name in new_field_names or field_name.startswith(' ')):
                        fields_to_delete.append(field_name)
                for row in data:
                    for field in fields_to_delete:
                        try:
                            del row[field]
                        except KeyError:
                            continue
                    for field in fields_to_change:
                        row[field] = fields_to_change[field]
                self.access_data = data
            if 'sorting' in new_structure:
                sorting_rules = new_structure['sorting']
                new_sorting = []
                for sort in sorting_rules:
                    if sort['name'] in new_field_names:
                        new_sorting.append(sort)
                if new_sorting:
                    new_structure['sorting'] = new_sorting
        self.title = new_structure['title']
        self.access_structure = new_structure
        self.structure_updated_at = timezone.now()


def update_fields(field, old_fields, old_field_names, fields_to_change):
    """Явно проставить значения полей грида, если они не указаны.

    Кусок кода из change_structure
    """
    field_name = field['name']
    # set default values
    if 'type' not in field:
        field['type'] = 'string'
    if field['type'] in ('select', 'staff'):
        if 'multiple' not in field:
            field['multiple'] = False
    if 'sorting' not in field:
        if field['type'] in SORTABLE_FIELD_TYPES:
            field['sorting'] = True
            if field['type'] in ('select', 'staff'):
                field['sorting'] = not field['multiple']
        else:
            field['sorting'] = False
    if 'required' not in field:
        field['required'] = False
    if field['type'] in ('staff', ticket_field_names.assignee, ticket_field_names.reporter) and 'format' not in field:
        field['format'] = 'i_first_name i_last_name'
    if field['type'] == 'number' and 'format' not in field:
        field['format'] = '%.2f'
    if field_name not in old_field_names:
        default_value = field.get('default', '')
        # правильнее было бы в зависимости от типа вставлять
        # нужные ключи, но для этого много нужно переписать функции,
        # формирующие контент ячеек в зависимости от типа в grids.utils.
        # Попробуем будет ли работать так, до этого тут был None или
        # default-значение
        fields_to_change[field_name] = {'raw': default_value}
    else:
        _description = old_fields[old_field_names.index(field_name)]
        new_type = field['type']
        old_type = _description.get('type', 'string')
        if old_type != new_type:
            # TODO: mess with different data types
            # NOW: delete value -P
            fields_to_change[field_name] = field.get('default')
