
import logging
from copy import deepcopy
from datetime import datetime, timedelta

from django.conf import settings
from wiki.legacy.choices import Choices

from wiki.grids.utils.base import HASH_KEY
from wiki.notifications.models import PageEvent
from wiki.pages.models import Revision
from wiki.utils import timezone
from wiki.utils.timer import track_time

structure_choices = Choices(COLUMN_CREATED=1, COLUMN_DELETED=2, COLUMN_MOVED=3, COLUMN_SORTS=4)

DEFAULT_TIMEOUT = timedelta(minutes=settings.GRID_PAGEEVENT_TIMEOUT)


def _get_raw_row(row):
    cells = dict((k, None if v is None else v['raw']) for k, v in row.items() if k.isdigit())
    return cells


def row_edited(row, user, grid, old_row, **kwargs):
    row_key = row[HASH_KEY]
    row_data = default_row_data(PageEvent.EVENT_TYPES.edit, row)
    row_data['old_row'] = old_row
    return row_changed(row, row_key, row_data, user, grid)


def row_created(row, user, grid):
    row_key = row[HASH_KEY]
    row_data = default_row_data(PageEvent.EVENT_TYPES.create, row)
    return row_changed(row, row_key, row_data, user, grid)


def row_deleted(row, user, grid):
    row_key = row[HASH_KEY]
    row_data = default_row_data(PageEvent.EVENT_TYPES.delete, row)
    return row_changed(row, row_key, row_data, user, grid)


def default_row_data(event_type, row, dt=None):
    """
    This is just a dumb helper
    """
    return {
        'type': event_type,
        'row': row,
        'datetime': dt or timezone.now().strftime('%Y-%m-%d %H-%M-%S'),
    }


@track_time
def row_changed(row, row_key, row_data, user, grid):
    """Create/update PageEvent, Revision if row has been edited, created, deleted
    @param row: the row
    @param row_key: row key
    @param row_data: dict returned by default_row_data()
    @param user: django User
    @param grid: the grid
    @see: http://wiki.yandex-team.ru/WackoWiki/dev/grid/versions
    """
    default_page_event_meta = {
        'data': {},
        'structure': {},
        'object_of_changes': 'data',
        'diff': {
            'added': {},
            'deleted': {},
        },
    }
    # Для того, чтобы не получилось ситуации, когда PageEvent выбирается командой notify, и изменяется здесь до того,
    # как notify его обработает и отправит (и проставит sent_at), не будет здесь выбирать объекты PageEvent, по которым
    # таймаут уже произошел или вот-вот произойдет. Заодно и лог изменений не будет содержать накопленные за очень
    # большое время ревизии.
    page_event_qs = (
        PageEvent.objects.select_related('author')
        .filter(
            page=grid,
            event_type=PageEvent.EVENT_TYPES.edit,
            timeout__gte=timezone.now() + timedelta(minutes=1),
            sent_at=None,
        )
        .order_by('-created_at', '-id')[:1]
    )

    data_changes = None
    if page_event_qs:
        page_event = page_event_qs[0]
        if page_event.meta.get('object_of_changes') == 'data' and page_event.author == user:
            data_changes = page_event
            page_event.timeout = timezone.now() + DEFAULT_TIMEOUT
    if data_changes is not None:
        if 'data' not in page_event.meta:
            meta = deepcopy(default_page_event_meta.copy)
        else:
            meta = deepcopy(page_event.meta)
        if row_key not in meta['data']:
            meta['data'][row_key] = [row_data]
        else:
            meta['data'][row_key].append(row_data)
    else:
        meta = default_page_event_meta.copy()
        meta['data'][row_key] = [row_data]
        page_event = PageEvent(
            event_type=PageEvent.EVENT_TYPES.edit,
            author=user,
            timeout=timezone.now() + DEFAULT_TIMEOUT,
            notify=True,
            page=grid,
        )

    grid.last_author = user

    revision = Revision.objects.filter(page=grid, mds_storage_id=grid.mds_storage_id).first()
    if not revision:
        revision = Revision.objects.create_from_page(grid)

    meta['revision_id'] = revision.id

    if row_data['type'] == PageEvent.EVENT_TYPES.create:
        raw_row = _get_raw_row(row)
        meta['diff']['added'][row[HASH_KEY]] = raw_row
    elif row_data['type'] == PageEvent.EVENT_TYPES.delete:
        raw_row = _get_raw_row(row)
        if row[HASH_KEY] not in meta['diff']['deleted']:
            meta['diff']['deleted'][row[HASH_KEY]] = raw_row
    else:
        new_raw_row = _get_raw_row(row)
        meta['diff']['added'][row[HASH_KEY]] = new_raw_row

        # Если строку редактируют не в первый раз, в диффе уже есть первоначальный вариант строки,
        # и заменять его не нужно.
        if row[HASH_KEY] not in meta['diff']['deleted']:
            old_raw_row = _get_raw_row(row_data['old_row'])
            meta['diff']['deleted'][row[HASH_KEY]] = old_raw_row

    page_event.meta = meta
    page_event.save()


row_change_action = {
    PageEvent.EVENT_TYPES.edit: row_edited,
    PageEvent.EVENT_TYPES.delete: row_deleted,
    PageEvent.EVENT_TYPES.create: row_created,
}


class UnproccessableEvent(Exception):
    pass


class CantGetRevision(UnproccessableEvent):
    def __init__(self, revision_id, *args, **kwargs):
        super(CantGetRevision, self).__init__(*args, **kwargs)
        logging.getLogger('django.request').warning('Unknown revision %s' % repr(revision_id))


class NoEventsToShow(UnproccessableEvent):
    def __init__(self, page_event_id, *args, **kwargs):
        super(NoEventsToShow, self).__init__(*args, **kwargs)


def changes_of_structure(user, grid, previous_structure):
    default_page_event_meta = {
        'structure': [],
        'object_of_changes': 'structure',
    }
    structure_data = {
        'structure': grid.access_structure,
        'structure_before': previous_structure,
        'datetime': datetime.now().strftime('%Y-%m-%d %H-%M-%S'),
    }
    page_event_qs = PageEvent.objects.filter(page=grid, event_type=PageEvent.EVENT_TYPES.edit, sent_at=None).order_by(
        '-created_at', '-id'
    )[:1]
    structure_changes = None
    if len(page_event_qs) > 0:
        page_event = page_event_qs[0]
        if page_event.meta.get('object_of_changes') == 'structure' and page_event.author == user:
            structure_changes = page_event
    if structure_changes is None:
        meta = default_page_event_meta.copy()
        page_event = PageEvent(
            event_type=PageEvent.EVENT_TYPES.edit,
            author=user,
            timeout=timezone.now() + DEFAULT_TIMEOUT,
            notify=True,
            page=grid,
        )
    else:
        page_event = structure_changes
        if 'structure' not in page_event.meta:
            meta = default_page_event_meta
        else:
            meta = page_event.meta.copy()
    meta['structure'].append(structure_data)
    # всегда создаем ревизию
    meta['revision_id'] = Revision.objects.create_from_page(grid).id
    page_event.meta = meta
    page_event.save()


def used_old_revision(grid, old_revision, user):
    """
    Обработать восстановление грида из старой ревизий:
        * создать PageEvent
        * создать ревизию
    """
    pe = PageEvent(
        event_type=PageEvent.EVENT_TYPES.edit,
        author=user,
        timeout=timezone.now() + DEFAULT_TIMEOUT,
        notify=True,
        page=grid,
    )
    new_revision = Revision.objects.create_from_page(grid)
    pe.meta = {
        'object_of_changes': 'old_revision',
        'revision_date': old_revision.created_at.strftime('%Y-%m-%d %H:%M:%S'),
        'revision_id': new_revision.id,
    }
    pe.save()
    return pe


def rollback_grid(grid, revision, user, commit=True):
    """
    Откатить грид `grid` на ревизию `revision`.
    Создает новую ревизию и PageEvent с помощью used_old_revision.
    """

    grid.body = revision.body
    grid.title = revision.title
    grid.last_author = user
    grid.modified_at = timezone.now()
    if commit:
        grid.save()

    used_old_revision(
        grid=grid,
        old_revision=revision,
        user=user,
    )
    return grid
