
from wiki.actions.classes.form_elements.mapper import build_form
from wiki.grids.logic.tickets import is_tracker_ticket_field
from wiki.grids.utils.base import HASH_KEY, FieldValidationError, filter_data_from_client, serialized_field
from wiki.utils import timezone


class RowValidationError(Exception):
    def __init__(self, errors, *args, **kwargs):
        super(RowValidationError, self).__init__(*args, **kwargs)
        self.errors = errors


def prepare_row(grid, row, request):
    """Подготовить ряд для вставки в грид и провалидировать его."""
    new_row = row.copy()
    for description in grid.access_structure['fields']:
        if not (
            description['name'] in new_row
            or
            # клиент не обновляет поля этого типа.
            # поэтому мы не записываем эти данные в грид.
            # Т.е. есть возможность что мы получим грид, у которого
            # в rows отсутствуют некоторые ячейки.
            # Однако при создании нового столбца из ticket_dependent_types
            # заполняются все ячейки этого столбца во всех строках.
            is_tracker_ticket_field(grid, description['name'])
        ):
            new_row[description['name']] = ''

    the_form = form(grid, new_row)
    form_errors = {}
    if the_form.is_valid():
        try:
            new_row = filter_data_from_client(grid, new_row, request)
        except FieldValidationError as inst:
            form_errors[inst.field_name] = inst.message
    else:
        for field in the_form:
            if field.errors:
                form_errors[field.name] = list(field.errors)
    if form_errors:
        raise RowValidationError(form_errors)

    return new_row


def insert_rows(grid, rows, request, after=None):
    """Правильная функция для вставки строк в грид.

    Grid, [{}, ... ], HttpRequest, string -> [string, string, ... ]

    @raises RowValidationError

    """
    if after:
        if after == 'last':
            row_number = len(grid.access_data)
        else:
            row_number = grid.access_idx[after] + 1
    else:
        row_number = 0

    return insert_rows_at_idx(grid, rows, request, row_number)


def insert_rows_at_idx(grid, rows, request, row_number):
    """
    @raises RowValidationError
    """

    appended = []
    data = grid.access_data
    meta = grid.access_meta
    hash = int(meta.get('autoincrement', 0))
    grid_dirty_at_row = row_number

    for row in rows:
        new_row = prepare_row(grid, row, request)
        hash += 1
        str_hash = str(hash)
        new_row[HASH_KEY] = str_hash
        data.insert(row_number, new_row)
        appended.append(str_hash)
        row_number += 1
    grid.access_data = data
    meta['autoincrement'] = hash
    grid._rebuild_index_at(grid_dirty_at_row)
    grid.access_meta = meta
    grid.modified_at = timezone.now()

    delattr(grid, serialized_field('body'))
    return appended


def edit_row(grid, new_row, row_id, request):
    old_row = grid.access_data[grid.access_idx[row_id]]
    form_row = new_row.copy()
    for field_name, value in old_row.items():
        if field_name.startswith('__'):
            continue
        if field_name not in new_row:
            # we must omit JIRA TICKETS for now
            # TODO: do something genious here!
            if not is_tracker_ticket_field(grid, field_name):
                if isinstance(value, dict):
                    form_row[field_name] = value.get('raw')

    the_form = form(grid, form_row)
    form_errors = {}
    if the_form.is_valid():
        try:
            new_row = filter_data_from_client(grid, new_row, request)
        except FieldValidationError as inst:
            form_errors[inst.field_name] = inst.message
    else:
        for field_name in the_form:
            if field_name.errors:
                form_errors[field_name.name] = list(field_name.errors)
    if form_errors:
        raise RowValidationError(form_errors)
    grid.change(row_id, new_row)


def edit_rows(grid, rows, request):
    for row_id in rows:
        edit_row(grid, rows[row_id], row_id, request)
    # invalidate the f_ng cache, yes!!
    # otherwise, the function "unserialize_data" is not called,
    # and missing tickets will not be created right away
    delattr(grid, serialized_field('body'))
    return list(rows.keys())


def form(grid, bind_fields=None):
    """
    Создать вики-форму по описанию структуры табличного списка.

    ВНИМАНИЕ: игнорирует поле required у табличных списков (WIKI-5838)

    @type grid: Grid
    @type bind_fields: dict
    @rtype: ClearForm
    """
    structure = grid.access_structure
    if 'fields' not in structure:
        structure['fields'] = []
    fields = {}
    form_description = {'fields': fields}
    for field in structure['fields']:
        # copy values
        form_field = {}
        for key in field:
            if key == 'name':
                continue
            form_field[str(key)] = field[key]
        fields[str(field['name'])] = form_field.copy()
        the_new_field = fields[str(field['name'])]
        the_new_field['required'] = False
        if the_new_field['type'] == 'select':
            the_new_field['options'] = []
            the_new_field['type'] = 'string'
        if bind_fields is None:
            continue
        if not bind_fields.get(field['name']):
            continue
        if the_new_field['type'] == 'ticket':
            data = bind_fields[field['name']]
            if 'raw' in data:
                bind_fields[field['name']] = data['raw']
    form = build_form(source=form_description, bind_fields=bind_fields, url=None)
    for description in structure['fields']:
        name = description['name']
        if description['type'] == 'date':
            form.fields[name].input_formats = ['%Y-%m-%d']
    return form


def are_cells_changed(server_row, user_row, data):
    """
    Сравниваем значения в ячейках строки со значениями в data.

    @rtype bool
    """
    for key, value in data.items():
        server_value = server_row.get(key, {}).get('raw')
        if server_value != value:
            # новое значение пользователя и значение на сервере не совпадают
            if server_value != user_row.get(key, {}).get('raw'):
                # и не совпадают в версии пользователя и на сервере
                # значит кто-то уже поменял
                return True

    return False
