import json

from django.conf import settings

from wiki.grids.logic.tickets import ticket_dependent_types
from wiki.grids.utils.tracker_ticket import load_ticket_fields_values


def serialized_field(field_name):
    return '__unserialized_' + field_name


class SerializeAsDictKey(object):
    def __init__(self, key, model_field, serialize, unserialize):
        self.key = key
        self.model_field = model_field
        self.serialize = serialize
        self.unserialize = unserialize

    def init_cache(self, obj):
        unserialized_field = serialized_field(self.model_field)
        if not hasattr(obj, unserialized_field):
            field_value = self.unserialize(getattr(obj, self.model_field))
            setattr(obj, unserialized_field, field_value)
        return getattr(obj, unserialized_field)[self.key]

    def __get__(self, obj, cls=None):
        unserialized = self.init_cache(obj)
        return unserialized

    def __set__(self, obj, value):
        unserialized_field = serialized_field(self.model_field)
        dict = getattr(obj, unserialized_field, {})
        dict[self.key] = value
        setattr(obj, self.model_field, self.serialize(dict))


def unserialize_json(text):
    if text is None:
        result = {}
    else:
        if text:
            result = json.loads(text)
        else:
            result = {}
    keys = ('structure', 'idx')
    for key in keys:
        if key not in result:
            result[key] = {}
    if 'data' not in result:
        result['data'] = []
    if 'meta' not in result:
        result['meta'] = {
            'autoincrement': 0,
        }
    return result


def unserialize_data(incomplete_json, numbers=None, hashes=None, user_auth=None):
    """Десериализовать только запрошенные строки.

    dict, [int, ...], [string, ...], HttpRequest -> list

    @param user_auth: UserAuth
    """
    json = make_beautiful_structure_from(incomplete_json)
    rows = []
    if numbers is not None:
        for n in numbers:
            rows.append(json['data'][n])
    if hashes is not None:
        for h in hashes:
            rows.append(json['data'][json['idx'][h]])
    json['data'] = rows

    grid_field_description = json['structure']['fields']

    number_fields = [field['name'] for field in grid_field_description if field['type'] == 'number']
    format_number_fields(json['data'], number_fields)

    if settings.ISSUE_TRACKERS_ENABLED:
        has_tickets = [desc for desc in grid_field_description if 'ticket' == desc.get('type')]
        if not has_tickets:
            return json['data']
        dependant_ticket_fields = dict(
            [(f['name'], f) for f in [f for f in grid_field_description if f.get('type') in ticket_dependent_types]]
        )
        has_tickets = [f['name'] for f in has_tickets]

        load_ticket_fields_values(
            ticket_fields=has_tickets,
            rows=json['data'],
            dependant_ticket_fields=dependant_ticket_fields,
            user_auth=user_auth,
        )

    return json['data']


def format_number_fields(rows, number_fields):
    """
    Преобразовать значения ячеек типа number в строковое представление для корректного отображения на клиенте.
    """
    for row in rows:
        for field in number_fields:
            field_value = row.get(field)
            if field_value and field_value.get('raw') is not None:
                if not field_value.get('sort'):
                    # если это старая ячейка (сохраненная до начала использования атрибута sort), то у нее атрибут sort
                    # отсутствует, а нам он нужен для правильной сортировки значений ячеек с типом number,
                    # несконвертированных в строку для представления.
                    row[field]['sort'] = field_value.get('raw')
                row[field]['raw'] = str(field_value.get('raw'))


def make_beautiful_structure_from(ugly_structure):
    """Specify missing fields with default values

    ugly_structure is JSON
    """
    json = unserialize_json(ugly_structure)
    beautiful_structure = json['structure']
    if 'done' not in beautiful_structure:
        beautiful_structure['done'] = False
    if 'fields' not in beautiful_structure:
        beautiful_structure['fields'] = []
    default_field_values = {
        'type': 'string',
        'required': False,
    }
    for description in beautiful_structure['fields']:
        for option in default_field_values:
            if option not in description:
                description[option] = default_field_values[option]
        if description['type'] == 'select':
            if 'multiple' not in description:
                description['multiple'] = False
        elif description['type'] == 'staff':
            if 'multiple' not in description:
                description['multiple'] = False
            if 'format' not in description:
                description['format'] = 'first_name last_name'
        elif description['type'] == 'date':
            if 'format' not in description:
                description['format'] = 'd.m.Y'
        elif description['type'] == 'number':
            if 'format' not in description:
                description['format'] = '%.2f'
    return json
