import logging

from django.utils.translation import ugettext_lazy
from wiki.grids.utils.base import ModelCache, StaffSyncer, ticket_field_names, ticket_statuses
from wiki.org import org_staff
from wiki.utils.tracker import get_startrek_issues
from wiki.utils.tracker.issue_processors.grid_issue_processor import GridIssueProcessor

logger = logging.getLogger(__name__)

FAILED_TO_FETCH = ugettext_lazy('grids:Failed to fetch ticket')

"""
Позволяет переопределить пользовательский порядок сортировки приоритетов и статусов, например:
priority_sort_map = {
    'trivial': '10',
    'minor': '20',
    'normal': '30',
    'critical': '40',
    'blocker': '50',
}

status_sort_map = {
    'new': '10',
    'open': '20',
    'in progress': '30',
    'resolved': '40',
    'testing': '50',
    'tested': '60',
    'released': '70',
    'closed': '80',
}
В тикете WIKI-14622 решили оставить алфавитный порядок, т.к. другой порядок сортировки менее очевиден
"""
priority_sort_map = {}
status_sort_map = {}


def normalize_ticket_id(ticket_id):
    return ticket_id.upper().strip()


def allow_default_label(func):
    """
    Если получить тикет не получилось, написать сообщение об ошибке.
    """

    def _acceptor(field_ticket, field_description, row, *args, **kwargs):
        if field_ticket['status'] == ticket_statuses.FAILED:
            row[field_description['name']] = {
                'view': str(FAILED_TO_FETCH),
                'raw': '',
                'sort': '',
            }
            if field_description.get('icononly'):
                row[field_description['name']]['view'] = ''
            return
        return func(field_ticket, field_description, row, *args, **kwargs)

    return _acceptor


def process_staff(field_name, label):
    @allow_default_label
    def process_assignee(field_ticket, field_description, row, issue_map, staff_syncer, **kwargs):
        raw_value = row.get(field_description['name']) or {}
        raw_value['raw'] = issue_map[normalize_ticket_id(field_ticket['raw'])][field_name]
        if raw_value['raw'] is None:
            raw_value['raw'] = []
            raw_value['view'] = str(label)
        else:
            try:
                staff_syncer.sync_raw(raw_value, field_description['format'], multiple=False)
            except Exception:
                logger.exception(f'Staff syncer failure on {raw_value}')
        row[field_description['name']] = raw_value

    return process_assignee


def process_iconic_tracker_type(field_name, empty_value_label='', sort_values_map=None):
    if sort_values_map is None:
        sort_values_map = {}

    @allow_default_label
    def do_process(field_ticket, field_description, row, issue_map, **kwargs):
        raw_value = row.get(field_description['name']) or {}
        the_issue = issue_map[normalize_ticket_id(field_ticket['raw'])]

        # What is happening here (let's take field 'status' for example).
        # 1) Look up 'status' in `the_issue`. If it exists and is None,
        # the parameter does not exist
        # 2) otherwise ignore 'status', whether it exists or not, andlook up
        # 'status_name' and 'status_icon' directly

        if field_name in the_issue and the_issue[field_name] is None:
            raw_value = {
                'raw': '',
                'view': '',
            }
        else:
            raw_value['raw'] = the_issue[field_name + '_name'] or empty_value_label
            raw_value['sort'] = sort_values_map.get(raw_value['raw'].lower()) or raw_value['raw']
            icon = ''
            if the_issue[field_name + '_icon']:
                icon = "<img class=\"b-ticket-icon\" src=\"%s\" alt=\"\" /> " % the_issue[field_name + '_icon']
            if field_description.get('icononly'):
                raw_value['view'] = "<span title='%s'>%s</span>" % (raw_value['raw'], icon)
            else:
                raw_value['view'] = raw_value['raw']
                if icon != '':
                    raw_value['view'] = icon + ' ' + raw_value['view']

        row[field_description['name']] = raw_value

    return do_process


def process_field_value(field_name, process_value_func):
    @allow_default_label
    def do_process(field_ticket, field_description, row, issue_map, **kwargs):
        raw_value = row.get(field_description['name']) or {}
        the_issue = issue_map[normalize_ticket_id(field_ticket['raw'])]
        issue_value = the_issue[field_name]
        row[field_description['name']] = process_value_func(raw_value, issue_value)

    return do_process


def process_text_value(raw_value, issue_value):
    if issue_value is None:
        raw_value = {'raw': '', 'view': ''}
    else:
        raw_value['raw'] = issue_value
        raw_value['view'] = str(issue_value)

    return raw_value


def process_datetime_value(raw_value, issue_value):
    if not issue_value:
        raw_value = {'raw': '', 'view': ''}
    else:
        raw_value['raw'] = issue_value.strftime('%Y-%m-%d')
        raw_value['view'] = issue_value.strftime('%Y-%m-%d')

    return raw_value


def process_list_value(raw_value, issue_value):
    if not issue_value:
        raw_value = {'raw': [], 'view': []}
    else:
        issue_value = ',<br>'.join(issue_value)
        raw_value['raw'] = issue_value
        raw_value['view'] = issue_value

    return raw_value


def default_strategy(*args, **kwargs):
    logger.critical('Trying to process field and no strategy supplied! called with args=%s, kwargs%s', args, kwargs)


field_processing_strategy = {
    ticket_field_names.assignee: process_staff('assignee', ugettext_lazy('grids:Unassigned')),
    ticket_field_names.reporter: process_staff('reporter', ''),
    ticket_field_names.priority: process_iconic_tracker_type('priority', 'Normal', priority_sort_map),
    ticket_field_names.status: process_iconic_tracker_type('status', sort_values_map=status_sort_map),
    ticket_field_names.type: process_iconic_tracker_type('type', 'Bug'),
    ticket_field_names.subject: process_field_value('summary', process_text_value),
    ticket_field_names.updated_at: process_field_value('modified_at', process_datetime_value),
    ticket_field_names.created_at: process_field_value('created_at', process_datetime_value),
    ticket_field_names.original_estimation: process_field_value('original_estimation', process_text_value),
    ticket_field_names.fix_versions: process_field_value('fix_versions', process_list_value),
    ticket_field_names.components: process_field_value('components', process_list_value),
    ticket_field_names.story_points: process_field_value('story_points', process_text_value),
}


def process_ticket_connected_fields(row, ticket_field, dependant_ticket_fields, issue_map, staff_syncer):
    """This routine proccesses the connected fields
    @param ticket_field: field of type "ticket"
    @param dependant_ticket_fields: map of names of dependent field
    """
    if not dependant_ticket_fields:
        return
    if ticket_field is None or not ticket_field['raw']:
        for field in dependant_ticket_fields:
            # TODO: when edition introduced, consider, here might be some info, that user has typed in
            row[field] = None
    else:
        for field, fieldinfo in dependant_ticket_fields.items():
            processor = field_processing_strategy.get(fieldinfo['type'], default_strategy)
            row[field] = {}
            raw_value = row[field].get('raw')
            if raw_value is not None:
                row['raw'] = raw_value
            processor(ticket_field, fieldinfo, row, issue_map, staff_syncer=staff_syncer)


def _warmup_ticket_cache(issue_map, staff_syncer):
    staff_fields = ('assignee', 'reporter')

    logins = set([])
    for issue in issue_map:
        for staff in staff_fields:
            if issue_map[issue][staff]:
                try:
                    logins.add(issue_map[issue][staff])
                except Exception:
                    logger.exception(f'Odd value in staff field: {issue_map[issue][staff]}')

    qs = org_staff().filter(login__in=logins)
    staff_syncer.cache = ModelCache(qs)
    staff_syncer.cache.storage = set(qs)


def load_ticket_fields_values(ticket_fields, rows, dependant_ticket_fields, user_auth):
    # assume there is only 1 ticket field
    ticket = ticket_fields[0]
    ticket_keys = []
    for row in rows:
        field = row.get(ticket)
        if field is not None and field.get('raw'):
            ticket_keys.append(normalize_ticket_id(field.get('raw', '')))

    tickets_map = get_startrek_issues(
        issue_keys=ticket_keys,
        user_auth=user_auth,
        issue_processor=GridIssueProcessor,
    )

    missing_issues = [key for key in ticket_keys if key not in tickets_map]
    staff_syncer = StaffSyncer()  # caching syncer
    _warmup_ticket_cache(tickets_map, staff_syncer)

    for row in rows:
        field = row.get(ticket)
        if field is None or field.get('raw') is None or field.get('raw') == '':
            # TODO: when connected fields are introduced, you must also consider this one
            for name in dependant_ticket_fields:
                row[name] = None
            continue
        key = field.get('raw')

        if key in missing_issues:
            field['status'] = ticket_statuses.FAILED
        elif key in tickets_map:
            field['status'] = ticket_statuses.NORMAL
        else:
            field['status'] = ticket_statuses.FAILED  # key != normalized key

        process_ticket_connected_fields(row, field, dependant_ticket_fields, tickets_map, staff_syncer)
