from datetime import datetime

from wiki.grids.utils.base import COLOR_KEY
from wiki.grids.utils.tracker_ticket import ticket_field_names


def plain_value(td_value):
    if not td_value:
        return ''
    return str(td_value.get('value') or td_value.get('raw'))


def grid_type_to_td(value, field, empty_string_value=True):
    """
    Makes value for printing from some grid cell data
    """
    if not value:
        return
    result = {'type': field['type'], 'sort': value.get('raw'), 'value': value.get('raw')}
    if field['type'] == 'staff':
        result.update({'sort': value.get('sort', ''), 'value': list(value.get('raw') or value.get('view', ''))})
        return result
    elif field['type'] in (ticket_field_names.assignee, ticket_field_names.reporter):
        result.update(
            {
                'sort': value.get('raw'),
                'value': [value.get('view')],
            }
        )
        if not value.get('view'):
            result['value'] = value.get('raw', [])
    elif field['type'] == 'select' and value is not None:
        result['value'] = value['raw']
    elif field['type'] in (ticket_field_names.priority, ticket_field_names.type, ticket_field_names.status):
        result['value'] = value['view']
        result['sort'] = value.get('sort', '')
    elif field['type'] == ticket_field_names.subject:
        result['sort'] = result['value'] = value['raw']
    elif field['type'] in ('date', ticket_field_names.created_at, ticket_field_names.updated_at):
        try:
            result['value'] = datetime.strptime(value, '%Y-%m-%d').date()
        except Exception:
            pass
    elif field['type'] == 'string':
        if empty_string_value:
            # Функция grid_as_table используется при импорте грида для формирования данных для превью
            # фронт пытается сделать JSON.parse над этими данными, когда-то в value был bemjson, и это
            # работало, после переезда на новый форматер у нас есть только сырая разметка, на которой
            # JSON.parse падает, оставляем это поле пустым
            result['value'] = ''
        result['sort'] = value.get('sort', value.get('raw', '')).strip()
    elif field['type'] == 'ticket':
        result['value'] = value.get('view', '')
        result['sort'] = value.get('raw', '')
    if isinstance(result.get('sort'), str):
        result['sort'] = result['sort'].lower()
    return result


def value_getter(row, index_of_column):
    """
    Для сортировки
    """
    if row[index_of_column]:
        value = row[index_of_column].get('sort', '')
        if isinstance(value, str):
            return value.strip()
        return value
    else:
        return None


def grid_as_table(grid_structure, grid_data, do_sort=False, empty_string_value=True):
    columns = list(grid_structure['fields'])
    column_descriptions = dict([(_['name'], _) for _ in columns])
    names = [_['name'] for _ in columns]
    rows = []
    if len(grid_data) > 0:
        row_len = len(names)

    for row in grid_data:
        new_row = []
        for key in names:
            if key in row:
                new_row.append(grid_type_to_td(row[key], column_descriptions[key], empty_string_value))
            else:
                new_row.append(None)

        if COLOR_KEY in row:
            new_row = ColourfulRow(new_row)
            new_row.has_color_ = row[COLOR_KEY]

        for i in range(row_len):
            if not isinstance(row.get(names[i], None), dict):
                continue
            colour = row[names[i]].get(COLOR_KEY)
            if colour is None:
                continue
            new_row[i] = ColourfulCell(new_row[i])
            new_row[i].has_color_ = colour
        rows.append(new_row)

    sorting = grid_structure.get('sorting')
    if do_sort and sorting:
        for sort in reversed(sorting):
            try:
                index_of_column = names.index(sort['name'])
                rows.sort(key=lambda row: str(value_getter(row, index_of_column)), reverse=(sort['type'] == 'desc'))
            except (ValueError):
                continue
    return columns, rows


class ColourfulRow(list):
    has_color_ = 'transparent'


class ColourfulCell(dict):
    has_color_ = 'transparent'


def grid_raw_body(grid):

    def _extract_value(cell):
        value = cell.get('value', ' ') if cell is not None else ' '
        if isinstance(value, list):
            value = ', '.join(value)
        return str(value).replace('|', '/')

    columns, rows = grid_as_table(grid.access_structure, grid.get_rows(None), do_sort=True, empty_string_value=False)
    body_rows = []
    body_rows.append('#|')
    body_rows.append('||' + '| '.join([column.get('title', '') for column in columns]) + '||')
    for row in rows:
        body_rows.append('||' + '| '.join([_extract_value(cell) for cell in row]) + '||')
    body_rows.append('|#')
    return '\n'.join(body_rows)
