from django.db import transaction
from pydantic import ValidationError
from typing import Any

from wiki.inline_grids.models import Grid
from wiki.api_v2.public.grids.exceptions import ColumnRequiredError, ExtraRowValue, RowValidationError
from wiki.api_v2.public.grids.schemas import DateColumnSchema, GridColumnSchema, column_type_schemas


@transaction.atomic
def create_grid(
    user,
    organization,
    title: str,
    page,
    structure: GridColumnSchema = None,
    rows: list[list[Any]] = None
) -> Grid:
    if structure:
        structure = structure.dict()
        for field in structure['fields']:
            field['type'] = field['type'].value
    else:
        structure = {'fields': []}

    grid = Grid.objects.create(
        title=title,
        page=page,
        structure=structure,
    )

    if rows is not None:
        add_rows(grid, rows)
    return grid


def add_rows(grid, rows):
    validate_rows(grid, rows)

    grid_rows = grid.rows
    max_id = max([row['id'] for row in grid_rows]) if grid_rows else 0

    for row in rows:
        max_id += 1
        grid_rows.append({'id': max_id, 'row': row})

    grid.raw_data = grid_rows
    grid.save()


def validate_rows(grid, rows):
    columns = grid.columns
    columns_number = len(columns)
    for row_idx, row in enumerate(rows):
        if len(row) > columns_number:
            # Нам передали лишние данные для которых не хватает колонок
            raise ExtraRowValue(details=f'row {row_idx + 1} has extra values')

        # В случае, если количество элементов в строке меньше количества колонок
        # нужно проверить, что оставшиеся колонки не являются обязательными
        required = [column['slug'] for column in columns[len(row):] if column['required']]
        if required:
            error_msg = 'columns "{}" are required' if len(required) > 1 else 'column "{}" is required'
            error_msg = error_msg.format(', '.join(required))
            raise ColumnRequiredError(details=error_msg)

        for col_idx, (column, row_value) in enumerate(zip(columns, row)):
            if row_value is None:
                if column['required']:
                    raise ColumnRequiredError(details='column "{} is required'.format(column['slug']))
                continue

            column_type = column['type']
            schema_class = column_type_schemas[column_type]

            try:
                validated_value = schema_class(value=row_value).value
                if issubclass(schema_class, DateColumnSchema):
                    validated_value = str(validated_value)
                rows[row_idx][col_idx] = validated_value
            except ValidationError:
                raise RowValidationError(details=f'value "{row_value}" is not "{column_type}"')
