import logging
from dataclasses import asdict
from json import dumps as json_dumps
from json import loads as json_loads
from typing import List, Dict

from django.conf import settings
from wiki.api_svc.acl.consts import DcInheritableAcl
from wiki.integrations.ms.base_client import BaseDocRetriever
from wiki.integrations.ms.exceptions import (
    EnsureAccessFailed,
    Ms365AccessDenied,
    Ms365BadDownloadUrl,
    Ms365BadFormat,
    Ms365BadLink,
    Ms365BadSignature,
    Ms365FinalizationFailed,
    Ms365GenericApiFailure,
    Ms365HtmlConvertError,
    Ms365NotFound,
)
from wiki.integrations.ms.schema import DataclassJSONEncoder, LastModifiedRequest
from wiki.utils import api_request
from wiki.utils.errors import (
    ApiRequestBadRequest,
    ApiRequestFailed,
    ApiRequestForbidden,
    ApiRequestNotFound,
    ApiRequestWorkerFatal,
)
from wiki.utils.tvm2 import get_service_ticket
from wiki.utils import json_ex

logger = logging.getLogger(__name__)


class DocRetrieverClient(BaseDocRetriever):
    def resolve_url(self, url):
        return self._call_retriever('api/v1/resolve_url', {'url': url})

    def create_new_document(self, path):
        return self._call_retriever('api/v1/create_new_document', {'path': path})

    def ensure_access(self, supertag, login, check_first=True):
        return self._call_retriever(
            'api/v1/acl/ensure_access', {'supertag': supertag, 'login': login, 'check_first': check_first}
        )

    def provision_page(self, acls: Dict[str, DcInheritableAcl]):
        acls = json_ex.cleanup({k: asdict(v) for k, v in acls.items()})
        return self._call_retriever('api/v1/acl/provision_page', {'acls': acls})

    def prepare_upload(self, path):
        return self._call_retriever('api/v1/prepare_upload', {'path': path})

    def finalize_upload(self, upload_url, signature):
        return self._call_retriever('api/v1/finalize_upload', {'upload_url': upload_url, 'signature': signature})

    def check_access(self, login, domain, namespace, sourcedoc):
        return self._call_retriever(
            'api/v1/check_access',
            {'document': {'domain': domain, 'namespace': namespace, 'sourcedoc': sourcedoc}, 'login': login},
        )

    def get_lastmodified(self, requests: List[LastModifiedRequest]):
        return self._call_retriever(
            'api/v1/batch/get_lastmodified', data=json_dumps({'requests': requests}, cls=DataclassJSONEncoder)
        )

    def get_download_url(self, driveitem):
        return self._call_retriever('api/v1/get_download_url', {'driveitem': driveitem})

    def docx_html(self, driveitem):
        return self._call_retriever('api/v1/docx_html', {'driveitem': driveitem})

    def xlsx_html(self, driveitem):
        return self._call_retriever('api/v1/xlsx_html', {'driveitem': driveitem})

    def pptx_html(self, driveitem):
        return self._call_retriever('api/v1/pptx_html', {'driveitem': driveitem})

    # ----------------

    def _call_retriever(self, endpoint, json=None, method='post', data=None):
        try:
            response = api_request.api_request(
                method=method,
                url=settings.DOC_RETRIEVER_ENDPOINT + endpoint,
                service_ticket=get_service_ticket(str(settings.DOC_RETRIEVER_ENDPOINT_TVM_CLIENT_ID)),
                user_ticket=self.user_ticket,
                json=json,
                data=data,
                headers={'Content-Type': 'application/json'},
                timeout=60,
                verify=False,
                retry_500=True,
                remote_name='DocRetriever',
            )
            return response.json()
        except ApiRequestBadRequest as e:
            self._handle_request_error(json_loads(e.data))
        except ApiRequestForbidden:
            raise Ms365AccessDenied()
        except ApiRequestNotFound:
            raise Ms365NotFound()
        except (ApiRequestWorkerFatal, ApiRequestFailed):
            logger.error(f'DocRetriever panic: {method} {endpoint} ({data or json})')
            raise Ms365GenericApiFailure()

    def _handle_request_error(self, error):
        """
        error = {
            'error_code': MS365_BAD_LINK
            'message': 'readable message'
        }
        """

        mapping = {
            'MS365_ACCESS_DENIED': (Ms365AccessDenied, False),
            'MS365_BAD_LINK': (Ms365BadLink, False),
            'MS365_BAD_FORMAT': (Ms365BadFormat, False),
            'MS365_FINALIZATION_FAILED': (Ms365FinalizationFailed, True),
            'BAD_SIGNATURE': (Ms365BadSignature, True),
            'BAD_DOCUMENT': (Ms365HtmlConvertError, True),
            'BAD_DOWNLOAD_URL': (Ms365BadDownloadUrl, True),
            'user_not_found': (EnsureAccessFailed, True),
            'user_not_found_in_ms': (EnsureAccessFailed, False),
            'page_not_found_in_wiki': (EnsureAccessFailed, True),
        }

        if 'error_code' not in error or error['error_code'] not in mapping:
            logger.error('Unknown response: %s' % error)
            raise Ms365GenericApiFailure()

        exception, should_log = mapping[error['error_code']]
        if should_log:
            logger.error('I got this error from Retriever: %s' % error)

        raise exception
