import base64
import logging
import re
from datetime import timedelta

import msal
import requests
from django.conf import settings
from django.urls import reverse
from django.utils import timezone

from wiki.integrations.ms.exceptions import (
    AppIsUnauthorized,
    AuthFailed,
    Ms365AccessDenied,
    MsApiBadRequest,
    MsApiRequestFailed,
    TokenExpired,
)
from wiki.integrations.ms.models import AppAuthorization
from wiki.utils.crypto_helpers import decrypt_string, encrypt_string, generate_fernet_key

logger = logging.getLogger(__name__)
SCOPES = [
    'Files.Read.All',
    'Files.ReadWrite',
    'User.Read',
]

salt = b'\xcb\x85F\xfa\xa9\x96\xc8rh"\x1e\x940\xa3\x87\x83'
KEY = generate_fernet_key(settings.MS365_KEY, salt)


def msapi_apicall(token, method='get', endpoint='me', data=None, params=None):
    attempt = 0
    MAX_ATTEMPTS = 3

    last_cause = ''
    while attempt < MAX_ATTEMPTS:
        attempt += 1
        try:
            resp = requests.request(
                method,
                endpoint if endpoint.startswith('http') else settings.MS365_GRAPH_ENDPOINT + endpoint,
                params=params,
                data=data,
                headers={'Authorization': 'Bearer %s' % token},
            )
        except (requests.exceptions.RequestException, requests.exceptions.Timeout) as e:
            last_cause = str(e)
            continue

        if resp.status_code == 200:
            return resp.json()

        if resp.status_code == 401:
            raise TokenExpired()

        if resp.status_code == 403:
            raise Ms365AccessDenied()

        last_cause = '[%s] %s' % (resp.status_code, resp.content)
        if 400 <= resp.status_code < 429:
            raise MsApiBadRequest(last_cause)

    raise MsApiRequestFailed(last_cause)


def msapi_apicall_ex(method='get', endpoint='me', data=None, params=None):
    auth = AppAuthorization.get_service_account()
    token = get_access_token_from_auth(auth)
    return msapi_apicall(token, method, endpoint, data, params)


def _build_msal_app(cache=None, authority=None):
    return msal.ConfidentialClientApplication(
        settings.MS365_APP_ID,
        authority=settings.MS365_AUTHORITY,
        client_credential=settings.MS365_CLIENT_SECRET,
        token_cache=cache,
    )


def refresh_access_token(auth):
    app = _build_msal_app()
    token = app.acquire_token_by_refresh_token(decrypt_string(auth.refresh_token.encode(), KEY), scopes=SCOPES)
    apply_token_response(auth, token)


def apply_token_response(auth, response):
    auth.refresh_token = encrypt_string(response['refresh_token'], KEY).decode()
    auth.access_token = encrypt_string(response['access_token'], KEY).decode()
    auth.expires_at = timezone.now() + timedelta(seconds=response['expires_in'])
    auth.refreshed_at = timezone.now()
    auth.webauth_required = False
    auth.rotate_state(save=False)
    auth.save()


def get_access_token(user):
    try:
        auth = AppAuthorization.objects.get(user=user)
    except AppAuthorization.DoesNotExist:
        raise AppIsUnauthorized()

    return get_access_token_from_auth(auth)


def get_access_token_from_auth(auth):
    if auth.webauth_required:
        raise AppIsUnauthorized()

    if timezone.now() + timedelta(seconds=30) > auth.expires_at:
        refresh_access_token(auth)

    return decrypt_string(auth.access_token.encode(), KEY)


def get_auth_url(request):
    auth, _ = AppAuthorization.objects.get_or_create(user=request.user)
    app = _build_msal_app()
    return app.get_authorization_request_url(
        SCOPES, state=auth.rotate_state(), redirect_uri=request.build_absolute_uri(reverse('frontend:ms365:callback'))
    )


def encode_shared_link(full_link):
    # https://docs.microsoft.com/en-us/graph/api/shares-get?view=graph-rest-1.0&tabs=http#encoding-sharing-urls
    b64 = base64.b64encode(full_link.encode()).decode().replace('\n', '')
    b64 = b64.replace('/', '_')
    b64 = b64.replace('+', '-')
    b64 = re.sub('(=)+$', '', b64)
    b64 = 'u!' + b64
    return b64


def acquire_token(request, state, code):
    try:
        auth = AppAuthorization.objects.get(oauth2_state=state, user=request.user, webauth_required=True)
    except AppAuthorization.DoesNotExist:
        logger.warn('OAuth fraud? %s state not found')
        raise AuthFailed()

    app = _build_msal_app()
    response = app.acquire_token_by_authorization_code(
        code, redirect_uri=request.build_absolute_uri(reverse('frontend:ms365:callback')), scopes=SCOPES
    )
    if 'error' in response:
        logger.warn('OAuth failed %s' % response)
        raise AuthFailed()
    apply_token_response(auth, response)


def check_api(auth):
    token = get_access_token_from_auth(auth)
    try:
        msapi_apicall(token, 'get', 'me')
    except (TokenExpired, MsApiBadRequest) as ex:
        logger.warn(ex)
        return False

    return True


def get_download_link(auth, share_link):
    token = get_access_token_from_auth(auth)
    encoded_link = encode_shared_link(share_link)
    resp = msapi_apicall(token, 'get', 'shares/%s/driveItem' % encoded_link)
    f = msapi_apicall(token, 'get', 'drives/%s/items/%s' % (resp['parentReference']['driveId'], resp['id']))
    return f['@microsoft.graph.downloadUrl'], f['file']['mimeType']
