from urllib.parse import urlencode

from wiki.api_v2.public.pages.schemas import CloudPageContentSchema
from wiki.integrations.consts import Ms365AclManagementType
from wiki.integrations.ms.consts import Ms365DocType
from wiki.integrations.ms.consts import SharepointCloudSrc
from wiki.integrations.ms.exceptions import EnsureAccessDenied
from wiki.integrations.ms.utils import get_doc_retriever_client
from wiki.pages.models import CloudPage


class MsOfficeCloudPageProvider(object):
    DEFAULT_EMBED_QUERY_PARAMS = {
        Ms365DocType.DOCX: {'wdStartOn': 1, 'wdEmbedCode': 0},
        Ms365DocType.XLSX: {'wdAllowInteractivity': 'True', 'wdDownloadButton': 'True', 'wdInConfigurator': 'True'},
        Ms365DocType.PPTX: {'wdAr': '1.7777777777777777'},
    }

    @classmethod
    def get_cloud_src(cls, cloud_page: CloudPage):
        return SharepointCloudSrc

    @classmethod
    def ensure_access(cls, cloud_page: CloudPage, user):
        if cloud_page.acl_management != Ms365AclManagementType.WIKI:
            raise EnsureAccessDenied()
        client = get_doc_retriever_client()
        client.ensure_access(cloud_page.page.supertag, user.username, True)

    @classmethod
    def to_apiv2_view(cls, cloud_page: CloudPage, layout='_layouts/15/Doc.aspx') -> CloudPageContentSchema:
        return CloudPageContentSchema.parse_obj(cls.to_frontend_view(cloud_page, layout))

    @classmethod
    def to_frontend_view(cls, cloud_page: CloudPage, layout='_layouts/15/Doc.aspx'):
        s = cloud_page.get_cloud_src()

        query_params = {
            'sourcedoc': s.embedding.sourcedoc,
        }

        page_type = s.document.type
        query_params.update(cls.DEFAULT_EMBED_QUERY_PARAMS[page_type])

        embed = {}
        for field_name, action_name in [('iframe_src', 'embedview'), ('edit_src', 'edit')]:
            query_params['action'] = action_name
            embed[field_name] = 'https://{tenant}{ns}{layout}?{qs}'.format(
                tenant=s.embedding.domain,
                ns=s.embedding.namespace,
                layout=layout,
                qs=urlencode(query_params),
            )

        acl_management = cloud_page.acl_management or Ms365AclManagementType.UNKNOWN
        data = {
            'embed': embed,
            'acl_management': acl_management.value,
            'type': page_type.value,
            'filename': s.document.filename,
        }

        return data

    @classmethod
    def get_download_url(cls, cloud_page):
        client = get_doc_retriever_client()
        driveitem = cloud_page.cloud_src.get('driveitem')
        client_response = client.get_download_url(driveitem)
        return client_response['url']
