
import re
from typing import List
from urllib.parse import parse_qs, urlparse

from django.conf import settings
from django.urls import reverse
from wiki.integrations.ms.base_client import BaseDocRetriever
from wiki.integrations.ms.consts import Ms365DocType
from wiki.integrations.ms.exceptions import Ms365BadLink, Ms365BadUploadSession
from wiki.integrations.ms.schema import LastModifiedRequest


class DocRetrieverStubClient(BaseDocRetriever):
    ALLOWED_NETLOCS = frozenset(['yandexteam.sharepoint.com', 'yandexteam-my.sharepoint.com'])

    NS_REGEX = re.compile(r'^(/:(?P<app>[wxp]):/r)?(?P<ns>[\w\d_\-/]*?)_layouts/(?P<layout_id>\d+)/Doc\.aspx$')

    VERB_MAPPER = {
        'w': Ms365DocType.DOCX,
        'x': Ms365DocType.XLSX,
        'p': Ms365DocType.PPTX,
    }

    DEFAULT_RESULT = {
        'driveitem': {
            'drive_id': 'b!R1UocfNXyU6bFmI8Bc5GDLERVSUjmrJKri57nSGRHnMpTM39UhheSJD3jlShBohv"',
            'item_id': '01CFXCC6GQ3SD3TP75RBHI53TDAILEMBEL',
        },
    }

    def ensure_access(self, supertag, login, check_first=True):
        pass

    def provision_page(self, acls):
        pass

    def create_new_document(self, url):
        result = self.DEFAULT_RESULT.copy()

        result['embedding'] = {
            'domain': 'yandexteam.sharepoint.com',
            'namespace': '/sites/wiki-dev/',
            'sourcedoc': '{B987DCD0-FDBF-4E88-8EEE-63021646048B}',
        }

        filename = url.split('/')[-1]

        result['document'] = {
            'type': Ms365DocType(filename.split('.')[-1]).value,
            'relative_path': '/my_files',
            'filename': filename,
        }

        return result

    def check_access(self, login, domain, namespace, sourcedoc):
        return {'access_level': 'read'}

    def resolve_url(self, url):
        result = self.DEFAULT_RESULT.copy()

        try:
            parts = urlparse(url)
        except Exception:
            raise Ms365BadLink()

        queryparams = parse_qs(parts.query)
        match = self.NS_REGEX.match(parts.path)

        if parts.netloc.lower() not in self.ALLOWED_NETLOCS:
            raise Ms365BadLink()

        if match is None:
            raise Ms365BadLink()

        result['embedding'] = {
            'domain': parts.netloc,
            'namespace': match.groupdict()['ns'],
            'sourcedoc': queryparams['sourcedoc'][0],
        }

        result['document'] = {
            'type': (self.VERB_MAPPER.get(match.groupdict()['app'], Ms365DocType.DOCX)).value,
            'relative_path': '/my_files',
            'filename': queryparams['file'][0] if 'file' in queryparams else 'doc.docx',
        }

        return result

    CORRECT_SIGNATURE = 'correct_signature'

    def prepare_upload(self, url):
        return {
            'upload_to': 'https://' + settings.API_WIKI_HOST + reverse('frontend:ms365:put_blackhole'),
            'signature': self.CORRECT_SIGNATURE,
        }

    def finalize_upload(self, upload_url, signature):
        if signature != self.CORRECT_SIGNATURE:
            raise Ms365BadUploadSession()
        return {
            'driveitem': {
                'drive_id': 'b!R1UocfNXyU6bFmI8Bc5GDLERVSUjmrJKri57nSGRHnMpTM39UhheSJD3jlShBohv"',
                'item_id': '01CFXCC6GQ3SD3TP75RBHI53TDAILEMBEL',
            },
            'document': {
                'type': 'pptx',
                'filename': 'Demo_Sprint_Kholodoque.pptx',
                'relative_path': '/My_Presetations',
            },
            'embedding': {
                'domain': 'yandexteam.sharepoint.com',
                'namespace': '/sites/wiki-dev/',
                'sourcedoc': '{B987DCD0-FDBF-4E88-8EEE-63021646048B}',
            },
        }

    def get_lastmodified(self, requests: List[LastModifiedRequest]):
        responses = {}
        for request in requests:
            responses[str(request.id)] = {'lastModifiedDateTime': '2020-09-11T09:11:34Z'}

        return {'responses': responses}
