import io
import logging
import re

import requests
from django.conf import settings
from django.core.files.base import ContentFile
from django.http import HttpResponseRedirect, JsonResponse
from django.views.generic import View
from django_replicated.utils import routers
from docx import Document
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.errors.rest_api_error import ResourceAlreadyExists
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_core.waffle_switches import MS365_INTEGRATION
from wiki.api_frontend.serializers.io import ok_response
from wiki.api_frontend.serializers.pages import PageEditionSerializer
from wiki.api_svc.acl.acl import get_acls
from wiki.api_svc.acl.is_managed_acl import get_acl_management
from wiki.async_process.serializers import AsyncRequestResultViewSerializer
from wiki.integrations.consts import Ms365AclManagementType
from wiki.integrations.ms.consts import DocCreationSourceType, Ms365DocType, SharepointCloudSrc
from wiki.integrations.ms.exceptions import (
    Ms365AccessDenied,
    Ms365AuthRequired,
    Ms365BadFormat,
    Ms365BadLink,
    Ms365BadUploadSession,
    Ms365GenericApiFailure,
    Ms365NotDocx,
)
from wiki.integrations.ms.models import AppAuthorization
from wiki.integrations.ms.msapi import acquire_token, check_api, get_auth_url, get_download_link
from wiki.integrations.ms.range_helpers import parse_http_range
from wiki.integrations.ms.serializers import (
    CheckAccessRequest,
    CreatePageRequest,
    FinalizeUploadRequest,
    ImportSharedDocxRequest,
    PreviewSharedDocxRequest,
    SetupServiceAccountRequest,
)
from wiki.integrations.ms.upload_intent import UploadIntentStorage
from wiki.integrations.ms.utils import (
    extract_presentation_params,
    extract_src_from_iframe,
    is_embed_iframe,
    parse_ms365_url,
    get_doc_retriever_client,
)
from wiki.pages.access import has_access, is_admin
from wiki.pages.logic.import_file import ImportException, import_file
from wiki.pages.logic.import_file.doc import _process_file
from wiki.pages.models import CloudPage, Page
from wiki.pages.utils.remove import delete_page
from wiki.utils.supertag import translit, normalize_supertag

logger = logging.getLogger(__name__)


class PreviewSharedDocx(WikiAPIView):
    FEATURE_FLAG = MS365_INTEGRATION

    serializer_class = PreviewSharedDocxRequest

    def _get_document(self, shared_link):
        auth = AppAuthorization.get_service_account()
        url, mime = get_download_link(auth, shared_link)

        if mime != 'application/vnd.openxmlformats-officedocument.wordprocessingml.document':
            raise Ms365NotDocx('Actual MIME was %s' % mime)

        data = requests.get(url)

        if data.status_code != 200:
            raise Ms365AccessDenied('[%s] %s' % (data.status_code, data.content))

        return data.content

    @raises(Ms365AuthRequired, Ms365AccessDenied, InvalidDataSentError, UserHasNoAccess)
    def post(self, request, *args, **kwargs):
        s = self.get_serializer(data=request.data)  # type: PreviewSharedDocxRequest
        s.is_valid(raise_exception=True)

        doc = Document(io.BytesIO(self._get_document(s.validated_data['shared_link'])))
        wt = _process_file(doc)

        return Response({'content': ''.join(wt)}, status=200)


class ImportSharedDocx(PreviewSharedDocx):
    FEATURE_FLAG = MS365_INTEGRATION

    serializer_class = ImportSharedDocxRequest

    def check_access(self, supertag):
        if not (is_admin(self.request.user) or has_access(supertag, self.request.user, privilege='write')):
            raise UserHasNoAccess

    @raises(Ms365AuthRequired, Ms365AccessDenied, InvalidDataSentError, UserHasNoAccess)
    def post(self, request, *args, **kwargs):
        s = self.get_serializer(data=request.data)  # type: ImportSharedDocxRequest
        s.is_valid(raise_exception=True)

        self.check_access(s.validated_data['supertag'])
        file_to_import = ContentFile(self._get_document(s.validated_data['shared_link']))
        file_to_import.name = 'temp.docx'

        request.tag = s.validated_data['supertag']
        request.supertag = s.validated_data['supertag']

        serializer = PageEditionSerializer(
            data={'title': 'Imported docx', 'body': ''}, context=self.get_serializer_context()
        )
        serializer.is_valid(raise_exception=True)
        page = serializer.save()
        try:
            self.request.page = page
            import_file(file_to_import, self.request)
        except ImportException as e:
            delete_page(page)  # Если не удалось импортировать файл, то удаляем созданную страницу
            raise InvalidDataSentError(str(e))
        except Exception:
            delete_page(page)  # Если не удалось импортировать файл, то удаляем созданную страницу
            raise

        return ok_response()


ALLOWED_MIME = {
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document': Ms365DocType.DOCX,
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet': Ms365DocType.XLSX,
    'application/vnd.openxmlformats-officedocument.presentationml.presentation': Ms365DocType.PPTX,
}


class CheckAccessView(WikiAPIView):
    serializer_class = CheckAccessRequest

    @raises(Ms365BadLink)
    def post(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        client = get_doc_retriever_client()
        url = serializer.validated_data['iframe_src']
        domain, ns, sourcedoc = parse_ms365_url(url)
        cloud_result = client.check_access(self.request.user.username, domain, ns, sourcedoc)
        return Response(cloud_result)


def trim_slashes(path: str):
    path = re.sub(r'^(/+)', '', path)
    path = re.sub(r'(/+)$', '', path)
    return path


def handle_from_url(client, url_or_iframe):
    # мы принимаем или урл на облачный документ, или embed который офис отдает
    # при "embed this excel sheet" -- он будет содержать еще размер и параметры как показывать
    # (presentation_params)

    presentation_params = {}

    if is_embed_iframe(url_or_iframe):
        url = extract_src_from_iframe(url_or_iframe)
        presentation_params = extract_presentation_params(url)
    else:
        url = url_or_iframe

    cloud_result = client.resolve_url(url)
    return cloud_result, presentation_params


def handle_empty_doc(client, desired_slug, doctype: Ms365DocType):
    url = '{}/{}.{}'.format(
        settings.MS365_WIKI_PREFIX,
        normalize_supertag(desired_slug),
        doctype.value,
    )

    return client.create_new_document(url)


def handle_upload_doc(client, desired_slug, doctype: Ms365DocType, data):
    url = '{}/{}.{}'.format(
        settings.MS365_WIKI_PREFIX,
        normalize_supertag(desired_slug),
        doctype.value,
    )

    cloud_result = client.prepare_upload(url)
    session_guid = UploadIntentStorage.store_intent(
        {'upload_to': cloud_result['upload_to'], 'signature': cloud_result['signature'], 'data': data}
    )
    return session_guid, cloud_result['upload_to']


def handle_finalize_upload(client, intent_guid):
    data = UploadIntentStorage.load_intent(intent_guid)

    url = data['upload_to']
    signature = data['signature']
    validated_data = data['data']

    cloud_result = client.finalize_upload(url, signature)
    UploadIntentStorage.delete_intent(intent_guid)
    return cloud_result, validated_data


def create_cloud_page(new_page: Page, cloud_result, presentation_params=None):
    mgmt = get_acl_management(SharepointCloudSrc.parse_obj(cloud_result))

    cloud_page = CloudPage(
        page=new_page, acl_management=mgmt, cloud_src=cloud_result, presentation_params=presentation_params or {}
    )
    cloud_page.save()

    if mgmt == Ms365AclManagementType.WIKI:
        client = get_doc_retriever_client()
        # noinspection PyBroadException
        try:
            client.provision_page(get_acls([new_page.supertag]))
        except Exception:
            logger.exception(f'Provision of page {new_page.supertag} FAILED; Setting unmanaged state')
            cloud_page.acl_management = Ms365AclManagementType.UNMANAGED
            cloud_page.save()

    return cloud_page


class BaseCreateCloudPageView(WikiAPIView):
    available_content_types = ['application/json']

    def check_write_access(self, supertag):
        if not (is_admin(self.request.user) or has_access(supertag, self.request.user, privilege='write')):
            raise UserHasNoAccess

    def check_page_exists(self, supertag):
        if Page.active.filter(supertag=supertag).exists():
            raise ResourceAlreadyExists()

    def _process_request(self, request, validated_data, source):

        request.tag = validated_data['supertag']
        request.supertag = translit(validated_data['supertag'])

        self.check_write_access(request.supertag)
        self.check_page_exists(request.supertag)

        client = get_doc_retriever_client()

        with client.user_identity(request.user_auth.tvm2_user_ticket):
            if source == DocCreationSourceType.FROM_URL:
                # 'url': str

                cloud_result, presentation_params = handle_from_url(client, validated_data['options']['url'])
                page = self._create_page(validated_data['title'], cloud_result, presentation_params)

                return self._prepare_response(page)

            if source == DocCreationSourceType.EMPTY_DOC:
                # 'doctype': Ms365DocType
                cloud_result = handle_empty_doc(
                    client, trim_slashes(validated_data['supertag']), validated_data['options']['doctype']
                )

                page = self._create_page(validated_data['title'], cloud_result)
                return self._prepare_response(page)

            if source == DocCreationSourceType.UPLOAD_DOC:
                # 'mime': str
                # ALLOWED_MIME[validated_data['options']['mime']]
                if validated_data['options']['mime'] not in ALLOWED_MIME:
                    raise Ms365BadFormat()

                session_guid, upload_to = handle_upload_doc(
                    client,
                    trim_slashes(validated_data['supertag']),
                    ALLOWED_MIME[validated_data['options']['mime']],
                    validated_data,
                )

                return Response({'upload_to': upload_to, 'upload_session': session_guid})

    def _create_page(self, title, cloud_result, presentation_params=None):
        page_edition_ser = PageEditionSerializer(
            data={'title': title, 'body': '', 'page_type': Page.TYPES.CLOUD},
            context=self.get_serializer_context(),
        )
        page_edition_ser.is_valid(raise_exception=True)
        new_page = page_edition_ser.save()
        create_cloud_page(new_page, cloud_result, presentation_params)
        return new_page

    def _prepare_response(self, new_page):
        return Response({'supertag': new_page.supertag})


class CreateCloudPageView(BaseCreateCloudPageView):
    serializer_class = CreatePageRequest

    @raises(
        Ms365BadLink,
        Ms365AccessDenied,
        Ms365GenericApiFailure,
        Ms365BadFormat,
        ResourceAlreadyExists,
        InvalidDataSentError,
        UserHasNoAccess,
        Ms365BadUploadSession,
    )
    def post(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        return self._process_request(request, serializer.validated_data, serializer.validated_data['source'])


class FinalizeUploadPageView(BaseCreateCloudPageView):
    serializer_class = FinalizeUploadRequest

    @raises(
        Ms365BadLink,
        Ms365AccessDenied,
        Ms365GenericApiFailure,
        Ms365BadFormat,
        ResourceAlreadyExists,
        InvalidDataSentError,
        UserHasNoAccess,
        Ms365BadUploadSession,
    )
    def post(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        guid = serializer.validated_data['upload_session']
        client = get_doc_retriever_client()

        with client.user_identity(request.user_auth.tvm2_user_ticket):
            cloud_result, validated_data = handle_finalize_upload(client, serializer.validated_data['upload_session'])

            request.tag = validated_data['supertag']
            request.supertag = translit(validated_data['supertag'])

            self.check_write_access(request.supertag)
            self.check_page_exists(request.supertag)

            new_page = self._create_page(validated_data['title'], cloud_result)
            UploadIntentStorage.delete_intent(guid)
            return self._prepare_response(new_page)


class AuthorizeMs365(WikiAPIView):
    FEATURE_FLAG = MS365_INTEGRATION

    serializer_class = AsyncRequestResultViewSerializer

    @raises()
    def get(self, request, *args, **kwargs):
        return HttpResponseRedirect(get_auth_url(request))


class SetupServiceAccount(WikiAPIView):
    FEATURE_FLAG = MS365_INTEGRATION

    serializer_class = SetupServiceAccountRequest

    @raises()
    def post(self, request, *args, **kwargs):
        acc = AppAuthorization.get_service_account()
        s = self.get_serializer(data=request.data)  # type: SetupServiceAccountRequest
        s.is_valid(raise_exception=True)
        force = s.validated_data['force']

        if not acc.webauth_required:
            if not force and check_api(acc):
                return Response(
                    data={
                        'action': 'account_ok',
                    },
                    status=200,
                )

        acc.webauth_required = True
        acc.user = request.user
        acc.save()
        return Response({'action': 'redirect', 'target': get_auth_url(request)}, status=200)


class HandleAuthCallbackMs365(WikiAPIView):
    FEATURE_FLAG = MS365_INTEGRATION

    serializer_class = AsyncRequestResultViewSerializer

    @raises()
    def get(self, request, *args, **kwargs):
        try:
            routers.use_state('master')
            acquire_token(request, code=request.GET.get('code'), state=request.GET.get('state'))
        finally:
            routers.revert()

        return ok_response()


class Blackhole(View):
    check_content_type_for = ('POST',)

    @raises()
    def put(self, request):
        http_range = request.META.get('HTTP_CONTENT_RANGE')

        if http_range is None:
            return JsonResponse({'error': 'Content-Range header is missing'}, status=400)
        if not parse_http_range(http_range):
            return JsonResponse({'error': 'Content-Range header is malformed'}, status=400)
        return JsonResponse({}, status=202)
