from django.conf import settings
import waffle
from wiki.api_core.waffle_switches import RETRIEVER_ACL, RETRIEVER_SEND_TYPE_AND_PAYLOAD
from wiki.cloudsearch.cloudsearch_client import only_switch
from django.db.models import Q

from wiki.cloudsearch.sqs_client import SqsClient
from wiki.integrations.consts import Ms365AclManagementType
from wiki.pages.models import Page, CloudPage


class RetrieverSqsClient(SqsClient):
    SQS_QUEUE_NAME = settings.SQS_RETRIEVER_QUEUE_NAME

    @only_switch(RETRIEVER_ACL)
    def on_model_update_acl(self, page: Page):
        pages = (
            CloudPage.objects.filter(
                page__org=page.get_model_org(), page__status=1, acl_management=Ms365AclManagementType.WIKI
            )
            .filter(Q(page=page) | Q(page__supertag__startswith=page.supertag + '/'))
            .select_related('page')
        )
        for child_page in pages:
            if waffle.switch_is_active(RETRIEVER_SEND_TYPE_AND_PAYLOAD):
                data = {'type': 'update_acl', 'payload': {'supertag': child_page.page.supertag}}
            else:
                data = {'supertag': child_page.page.supertag}
            self._send_message(data=data)

    @only_switch(RETRIEVER_SEND_TYPE_AND_PAYLOAD)
    def on_model_delete(self, page: Page):
        if page.page_type == Page.TYPES.CLOUD:
            payload = page.cloudpage.get_cloud_src().embedding.dict() | {'supertag': page.supertag}
            self._send_message(data={'type': 'delete', 'payload': payload})


RETRIEVER_SQS_CLIENT = RetrieverSqsClient()
